<?php
require_once '../config/db.php';
require_once '../config/functions.php';

requerirAutenticacion();
requerirRol(['admin', 'auxiliar']);

$anio_lectivo_id = getAnioLectivoSeleccionado();
$anio_lectivo_nombre = $_SESSION['anio_lectivo_nombre'] ?? '';

$page_title = "Reportes de Asistencia - SICA";
?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?= $page_title ?></title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.0/font/bootstrap-icons.css" rel="stylesheet">
    <link href="../assets/css/sica.css?v=5.2" rel="stylesheet">
</head>
<body>
    <?php include '../includes/layout_elems.php'; ?>

    <div class="d-flex justify-content-between align-items-center mb-4">
        <div class="d-flex align-items-center">
            <button class="btn btn-link d-lg-none me-3" onclick="toggleSidebarMobile()">
                <i class="bi bi-list fs-4"></i>
            </button>
            <div>
                <h3 class="mb-1">Reportes de Asistencia</h3>
                <p class="text-muted mb-0">Genera reportes individuales de asistencia por estudiante</p>
            </div>
        </div>
        <div class="d-flex gap-2">
            <button class="btn btn-outline-primary" onclick="window.print()">
                <i class="bi bi-printer me-1"></i> Imprimir
            </button>
            <a href="/SICA/dashboard.php" class="btn btn-outline-secondary">
                <i class="bi bi-house me-1"></i> Volver
            </a>
        </div>
    </div>

    <!-- Card de búsqueda -->
    <div class="card mb-4">
        <div class="card-header bg-primary text-white">
            <h5 class="mb-0"><i class="bi bi-search me-2"></i>Buscar Estudiante</h5>
        </div>
        <div class="card-body">
            <div class="row">
                <div class="col-md-8">
                    <label class="form-label fw-bold">Buscar por DNI, Apellidos o Nombres</label>
                    <div class="input-group">
                        <span class="input-group-text"><i class="bi bi-search"></i></span>
                        <input type="text" class="form-control" id="busquedaEstudiante"
                               placeholder="Escriba al menos 3 caracteres para buscar..."
                               oninput="buscarEstudiante()">
                    </div>
                </div>
                <div class="col-md-4">
                    <label class="form-label fw-bold">Año Lectivo</label>
                    <input type="text" class="form-control" value="<?= htmlspecialchars($anio_lectivo_nombre) ?>" disabled>
                </div>
            </div>
        </div>
    </div>

    <!-- Resultados de búsqueda -->
    <div id="resultadosContainer" style="display: none;">
        <div class="card">
            <div class="card-header bg-white">
                <div class="d-flex justify-content-between align-items-center">
                    <h5 class="mb-0">
                        <i class="bi bi-people me-2"></i>Resultados de Búsqueda
                    </h5>
                    <span class="badge bg-secondary" id="totalResultados">0 estudiantes</span>
                </div>
            </div>
            <div class="card-body">
                <div class="table-responsive">
                    <table class="table table-hover">
                        <thead>
                            <tr>
                                <th>DNI</th>
                                <th>Apellidos y Nombres</th>
                                <th>Nivel</th>
                                <th>Grado</th>
                                <th>Sección</th>
                                <th>Turno</th>
                                <th>Acciones</th>
                            </tr>
                        </thead>
                        <tbody id="tablaEstudiantes">
                            <tr>
                                <td colspan="7" class="text-center text-muted">
                                    No se encontraron resultados
                                </td>
                            </tr>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
    </div>

    <!-- Loading -->
    <div id="loadingIndicator" class="text-center py-5" style="display: none;">
        <div class="spinner-border text-primary" role="status">
            <span class="visually-hidden">Cargando...</span>
        </div>
        <p class="mt-3 text-muted">Buscando estudiantes...</p>
    </div>

    <!-- Instrucciones -->
    <div id="instruccionesContainer" class="card">
        <div class="card-body text-center py-5">
            <i class="bi bi-lightbulb text-warning" style="font-size: 4rem;"></i>
            <h4 class="mt-3">¿Cómo generar un reporte?</h4>
            <p class="text-muted mb-4">Sigue estos pasos para generar el reporte de asistencia de un estudiante:</p>
            <div class="row text-start">
                <div class="col-md-4 mb-3">
                    <div class="card h-100 border-primary">
                        <div class="card-body">
                            <h6 class="card-title text-primary">
                                <i class="bi bi-1-circle-fill me-2"></i>Paso 1
                            </h6>
                            <p class="card-text small">Escribe el DNI, apellidos o nombres del estudiante en el campo de búsqueda (mínimo 3 caracteres).</p>
                        </div>
                    </div>
                </div>
                <div class="col-md-4 mb-3">
                    <div class="card h-100 border-primary">
                        <div class="card-body">
                            <h6 class="card-title text-primary">
                                <i class="bi bi-2-circle-fill me-2"></i>Paso 2
                            </h6>
                            <p class="card-text small">Selecciona al estudiante de la lista de resultados. Se mostrarán sus datos básicos y asignación.</p>
                        </div>
                    </div>
                </div>
                <div class="col-md-4 mb-3">
                    <div class="card h-100 border-primary">
                        <div class="card-body">
                            <h6 class="card-title text-primary">
                                <i class="bi bi-3-circle-fill me-2"></i>Paso 3
                            </h6>
                            <p class="card-text small">Haz clic en "Generar Reporte" para ver el reporte completo con estadísticas, alertas e histórico de asistencias.</p>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <?php include '../includes/layout_scripts.php'; ?>

    <script>
        const ANIO_LECTIVO_ID = <?php echo $anio_lectivo_id; ?>;
        let timeoutBusqueda = null;

        function buscarEstudiante() {
            const busqueda = document.getElementById('busquedaEstudiante').value.trim();

            // Limpiar timeout anterior
            if (timeoutBusqueda) {
                clearTimeout(timeoutBusqueda);
            }

            // Ocultar si tiene menos de 3 caracteres
            if (busqueda.length < 3) {
                document.getElementById('resultadosContainer').style.display = 'none';
                document.getElementById('instruccionesContainer').style.display = 'block';
                return;
            }

            // Esperar 300ms antes de buscar (debounce)
            timeoutBusqueda = setTimeout(function() {
                ejecutarBusqueda(busqueda);
            }, 300);
        }

        async function ejecutarBusqueda(busqueda) {
            // Mostrar loading
            document.getElementById('loadingIndicator').style.display = 'block';
            document.getElementById('resultadosContainer').style.display = 'none';
            document.getElementById('instruccionesContainer').style.display = 'none';

            try {
                const response = await fetch(`ajax_reportes.php?accion=buscar_estudiantes&busqueda=${encodeURIComponent(busqueda)}&id_anio_lectivo=${ANIO_LECTIVO_ID}`);
                const data = await response.json();

                if (!data.success) {
                    throw new Error(data.mensaje);
                }

                mostrarResultados(data.estudiantes);
            } catch (error) {
                console.error('Error:', error);
                alert('Error al buscar estudiantes: ' + error.message);
                document.getElementById('instruccionesContainer').style.display = 'block';
            } finally {
                document.getElementById('loadingIndicator').style.display = 'none';
            }
        }

        function mostrarResultados(estudiantes) {
            const tbody = document.getElementById('tablaEstudiantes');
            document.getElementById('totalResultados').textContent = estudiantes.length + ' estudiantes';

            if (estudiantes.length === 0) {
                tbody.innerHTML = `
                    <tr>
                        <td colspan="7" class="text-center text-muted">
                            <i class="bi bi-search me-2"></i>No se encontraron estudiantes con ese criterio de búsqueda
                        </td>
                    </tr>
                `;
            } else {
                tbody.innerHTML = estudiantes.map(est => {
                    const botonDisabled = !est.tiene_programacion ? 'disabled' : '';
                    const botonTitle = !est.tiene_programacion ? 'El estudiante no tiene programación en el año lectivo actual' : 'Generar reporte individual de asistencia';

                    return `
                        <tr>
                            <td><strong>${est.dni}</strong></td>
                            <td>${est.apellido_paterno} ${est.apellido_materno}, ${est.nombres}</td>
                            <td>${est.nivel || '-'}</td>
                            <td>${est.grado || '-'}</td>
                            <td>${est.seccion || '-'}</td>
                            <td>${est.turno || '-'}</td>
                            <td>
                                <a href="reporte_individual.php?id=${est.id}"
                                   class="btn btn-sm btn-primary"
                                   ${botonDisabled}
                                   title="${botonTitle}">
                                    <i class="bi bi-file-earmark-text me-1"></i>Generar Reporte
                                </a>
                            </td>
                        </tr>
                    `;
                }).join('');
            }

            document.getElementById('resultadosContainer').style.display = 'block';
        }

        // Permitir búsqueda con Enter
        document.getElementById('busquedaEstudiante').addEventListener('keypress', function(e) {
            if (e.key === 'Enter') {
                e.preventDefault();
                buscarEstudiante();
            }
        });
    </script>
</body>
</html>
