<?php
/**
 * AJAX Reportes - Endpoint para reportes de asistencias
 * SICA - Sistema Integrado de Convivencia y Asistencia
 */

require_once '../config/db.php';
require_once '../config/functions.php';

requerirAutenticacion();

// Capturar errores de PHP y convertirlos a JSON
error_reporting(E_ALL);
ini_set('display_errors', 0);

function error_handler($errno, $errstr, $errfile, $errline) {
    json_response([
        'success' => false,
        'mensaje' => "Error PHP: $errstr",
        'detalle' => "Archivo: $errfile, Línea: $errline"
    ], 500);
}

set_error_handler('error_handler');

function exception_handler($exception) {
    json_response([
        'success' => false,
        'mensaje' => $exception->getMessage(),
        'archivo' => $exception->getFile(),
        'linea' => $exception->getLine()
    ], 500);
}

set_exception_handler('exception_handler');

$accion = $_GET['accion'] ?? $_POST['accion'] ?? '';

try {
    switch ($accion) {
        case 'buscar_estudiantes':
            buscar_estudiantes();
            break;

        case 'buscar_secciones':
            buscar_secciones();
            break;

        case 'obtener_reporte_individual':
            obtener_reporte_individual();
            break;

        case 'obtener_reporte_seccion':
            obtener_reporte_seccion();
            break;

        case 'obtener_grilla_seccion':
            obtener_grilla_seccion();
            break;

        case 'obtener_estadisticas_generales':
            obtener_estadisticas_generales();
            break;

        case 'obtener_alertas_estudiante':
            obtener_alertas_estudiante();
            break;

        default:
            json_response(['success' => false, 'mensaje' => 'Acción no válida'], 400);
    }
} catch (Exception $e) {
    json_response(['success' => false, 'mensaje' => 'Error: ' . $e->getMessage()], 500);
}

/**
 * Busca estudiantes por DNI, apellidos o nombres
 * Filtra por año lectivo y estado activo
 */
function buscar_estudiantes() {
    $busqueda = $_GET['busqueda'] ?? '';
    $anio_lectivo_id = $_GET['id_anio_lectivo'] ?? getAnioLectivoSeleccionado();

    if (strlen($busqueda) < 3) {
        json_response([
            'success' => false,
            'mensaje' => 'La búsqueda debe tener al menos 3 caracteres'
        ], 400);
    }

    // Buscar estudiantes con su programación
    $busqueda_param = "%{$busqueda}%";

    $estudiantes_result = query("
        SELECT DISTINCT
            e.id,
            e.dni,
            e.apellido_paterno,
            e.apellido_materno,
            e.nombres,
            e.genero,
            e.estado,
            CONCAT(e.apellido_paterno, ' ', e.apellido_materno, ', ', e.nombres) as nombre_completo,
            n.nombre as nivel,
            g.grado,
            s.seccion,
            t.nombre as turno,
            CASE WHEN pe.id IS NOT NULL THEN 1 ELSE 0 END as tiene_programacion
        FROM estudiantes e
        LEFT JOIN programaciones_estudiantes pe ON e.id = pe.id_estudiante
            AND pe.estado = 'activo'
        LEFT JOIN programaciones p ON pe.id_programacion = p.id
            AND p.id_anio_lectivo = ?
            AND p.estado = 'activo'
        LEFT JOIN secciones s ON p.id_seccion = s.id
        LEFT JOIN grados g ON s.id_grado = g.id
        LEFT JOIN niveles n ON g.id_nivel = n.id
        LEFT JOIN turnos t ON p.id_turno = t.id
        WHERE e.estado = 'activo'
          AND (
              e.dni LIKE ?
              OR e.apellido_paterno LIKE ?
              OR e.apellido_materno LIKE ?
              OR e.nombres LIKE ?
              OR CONCAT(e.apellido_paterno, ' ', e.apellido_materno) LIKE ?
              OR CONCAT(e.apellido_paterno, ' ', e.apellido_materno, ', ', e.nombres) LIKE ?
          )
        ORDER BY e.apellido_paterno, e.apellido_materno, e.nombres
        LIMIT 20
    ", array_merge([$anio_lectivo_id], array_fill(0, 6, $busqueda_param)));

    if (!$estudiantes_result) {
        json_response([
            'success' => false,
            'mensaje' => 'Error al buscar estudiantes'
        ], 500);
    }

    $estudiantes = $estudiantes_result->fetchAll();

    json_response([
        'success' => true,
        'estudiantes' => $estudiantes,
        'total' => count($estudiantes),
        'busqueda' => $busqueda,
        'anio_lectivo_id' => $anio_lectivo_id
    ]);
}

/**
 * Obtiene el reporte individual de asistencia de un estudiante
 */
function obtener_reporte_individual() {
    $id_estudiante = $_GET['id_estudiante'] ?? null;
    $fecha_inicio = $_GET['fecha_inicio'] ?? null;
    $fecha_fin = $_GET['fecha_fin'] ?? null;
    $anio_lectivo_id = getAnioLectivoSeleccionado();

    if (!$id_estudiante) {
        json_response(['success' => false, 'mensaje' => 'Falta el ID del estudiante'], 400);
    }

    // Si no se proporcionan fechas, usar todo el año lectivo
    if (!$fecha_inicio || !$fecha_fin) {
        $anio_result = query("SELECT fecha_inicio, fecha_fin FROM anios_lectivos WHERE id = ?", [$anio_lectivo_id]);
        if (!$anio_result) {
            json_response(['success' => false, 'mensaje' => 'No se encontró el año lectivo'], 404);
        }
        $anio_info = $anio_result->fetch();
        $fecha_inicio = $anio_info['fecha_inicio'];
        $fecha_fin = $anio_info['fecha_fin'];
    }

    // Obtener información del estudiante con su programación, tutor, auxiliar y apoderado
    $estudiante_result = query("
        SELECT
            e.id,
            e.dni,
            e.apellido_paterno,
            e.apellido_materno,
            e.nombres,
            e.genero,
            e.estado,
            e.telefono as estudiante_telefono,
            e.direccion,
            CONCAT(e.apellido_paterno, ' ', e.apellido_materno, ', ', e.nombres) as nombre_completo,
            n.nombre as nivel,
            g.grado,
            s.seccion,
            t.nombre as turno,
            t.abreviatura as turno_abreviatura,
            an.anio as anio_lectivo,
            p.id_anio_lectivo as programacion_anio_id,
            p.id as programacion_id,
            doc_tutor.dni as tutor_dni,
            doc_tutor.apellido_paterno as tutor_apellido_paterno,
            doc_tutor.apellido_materno as tutor_apellido_materno,
            doc_tutor.nombres as tutor_nombres,
            doc_tutor.email as tutor_email,
            doc_tutor.telefono as tutor_telefono,
            CONCAT(doc_tutor.apellido_paterno, ' ', doc_tutor.apellido_materno, ', ', doc_tutor.nombres) as tutor_nombre,
            doc_auxiliar.dni as auxiliar_dni,
            doc_auxiliar.apellido_paterno as auxiliar_apellido_paterno,
            doc_auxiliar.apellido_materno as auxiliar_apellido_materno,
            doc_auxiliar.nombres as auxiliar_nombres,
            doc_auxiliar.telefono as auxiliar_telefono,
            CONCAT(doc_auxiliar.apellido_paterno, ' ', doc_auxiliar.apellido_materno, ', ', doc_auxiliar.nombres) as auxiliar_nombre,
            apo.dni as apoderado_dni,
            apo.apellido_paterno as apoderado_apellido_paterno,
            apo.apellido_materno as apoderado_apellido_materno,
            apo.nombres as apoderado_nombres,
            apo.email as apoderado_email,
            apo.telefono as apoderado_telefono,
            apo.telefono_alternativo as apoderado_telefono_alt,
            CONCAT(apo.apellido_paterno, ' ', apo.apellido_materno, ', ', apo.nombres) as apoderado_nombre
        FROM estudiantes e
        INNER JOIN programaciones_estudiantes pe ON e.id = pe.id_estudiante
        INNER JOIN programaciones p ON pe.id_programacion = p.id
        INNER JOIN secciones s ON p.id_seccion = s.id
        INNER JOIN grados g ON s.id_grado = g.id
        INNER JOIN niveles n ON g.id_nivel = n.id
        INNER JOIN turnos t ON p.id_turno = t.id
        INNER JOIN anios_lectivos an ON p.id_anio_lectivo = an.id
        LEFT JOIN docentes doc_tutor ON p.id_tutor = doc_tutor.id
        LEFT JOIN auxiliares doc_auxiliar ON p.id_auxiliar = doc_auxiliar.id
        LEFT JOIN estudiantes_apoderados ea ON e.id = ea.id_estudiante
            AND ea.id_anio_lectivo = ?
            AND ea.estado = 'activo'
        LEFT JOIN apoderados apo ON ea.id_apoderado = apo.id
        WHERE e.id = ?
          AND p.id_anio_lectivo = ?
          AND pe.estado = 'activo'
          AND p.estado = 'activo'
    ", [$anio_lectivo_id, $id_estudiante, $anio_lectivo_id]);

    if (!$estudiante_result) {
        json_response([
            'success' => false,
            'mensaje' => 'Error al consultar la programación del estudiante',
            'detalle' => 'El estudiante no tiene una programación activa en el año lectivo seleccionado',
            'id_estudiante' => $id_estudiante,
            'anio_lectivo_id' => $anio_lectivo_id
        ], 404);
    }

    $estudiante = $estudiante_result->fetch();

    if (!$estudiante) {
        json_response([
            'success' => false,
            'mensaje' => 'Estudiante no encontrado o sin programación activa',
            'detalle' => "El estudiante con ID $id_estudiante no tiene una programación activa en el año lectivo con ID $anio_lectivo_id. Por favor, asígnelo a una sección en el módulo de Programaciones.",
            'id_estudiante' => $id_estudiante,
            'anio_lectivo_id' => $anio_lectivo_id
        ], 404);
    }

    // Obtener todas las asistencias del periodo
    $asistencias_result = query("
        SELECT
            a.id,
            a.fecha,
            a.hora_registro,
            a.minutos_tardanza,
            a.observaciones,
            a.registrado_por,
            a.metodo_registro,
            a.estado_codigo,
            ea.nombre as estado_nombre,
            ea.abreviatura as estado_abreviatura,
            ea.color as estado_color,
            ea.icono as estado_icono,
            ea.conteo_asistencia,
            ea.permite_tardanza
        FROM asistencias a
        INNER JOIN estados_asistencia ea ON a.id_estado_asistencia = ea.id
        WHERE a.id_estudiante = ?
          AND a.id_anio_lectivo = ?
          AND a.fecha BETWEEN ? AND ?
        ORDER BY a.fecha DESC
    ", [$id_estudiante, $anio_lectivo_id, $fecha_inicio, $fecha_fin]);

    if (!$asistencias_result) {
        json_response([
            'success' => false,
            'mensaje' => 'Error al consultar asistencias',
            'debug' => [
                'id_estudiante' => $id_estudiante,
                'anio_lectivo_id' => $anio_lectivo_id,
                'fecha_inicio' => $fecha_inicio,
                'fecha_fin' => $fecha_fin
            ]
        ], 500);
    }

    $asistencias = $asistencias_result->fetchAll();

    // Calcular estadísticas
    $total_dias = count($asistencias);
    $total_asistencias = 0;
    $total_tardanzas = 0;
    $total_faltas_justificadas = 0;
    $total_faltas_injustificadas = 0;
    $total_minutos_tardanza = 0;
    $dias_asistidos = 0;

    foreach ($asistencias as $a) {
        $total_minutos_tardanza += $a['minutos_tardanza'] ?? 0;

        // Normalizar conteo_asistencia a booleano
        // Acepta: 'si', 'sí', 'yes', 'true', '1', true, 1
        $cuenta_como_asistencia = $a['conteo_asistencia'] ?? null;
        $cuenta = false;

        if (in_array(strtolower($cuenta_como_asistencia), ['si', 'sí', 'yes', 'true', '1'], true)) {
            $cuenta = true;
        } elseif ($cuenta_como_asistencia === true || $cuenta_como_asistencia === 1) {
            $cuenta = true;
        }

        if ($cuenta) {
            $dias_asistidos++;
        }

        switch ($a['estado_codigo']) {
            case 'Asistencia':
                $total_asistencias++;
                break;
            case 'Tardanza':
            case 'Tardanza Justificada':
                $total_tardanzas++;
                break;
            case 'Falta Justificada':
                $total_faltas_justificadas++;
                break;
            case 'Falta Injustificada':
                $total_faltas_injustificadas++;
                break;
        }
    }

    // Calcular porcentajes
    $porcentaje_asistencia = $total_dias > 0 ? round(($dias_asistidos / $total_dias) * 100, 1) : 0;
    $porcentaje_faltas_injustificadas = $total_dias > 0 ? round(($total_faltas_injustificadas / $total_dias) * 100, 1) : 0;

    // Detectar alertas
    $alertas = detectar_alertas(
        $porcentaje_asistencia,
        $porcentaje_faltas_injustificadas,
        $total_tardanzas,
        $total_faltas_injustificadas,
        $asistencias
    );

    // Obtener parámetros de la institución
    $cfg_result = query("SELECT parametro, valor FROM configuracion WHERE parametro IN ('nombre_institucion', 'logo_institucion')");
    $config = [];
    while ($row = $cfg_result->fetch()) {
        $config[$row['parametro']] = $row['valor'];
    }

    // Formatear respuesta
    json_response([
        'success' => true,
        'data' => [
            'institucion' => [
                'nombre' => $config['nombre_institucion'] ?? 'SICA',
                'logo' => $config['logo_institucion'] ?? ''
            ],
            'estudiante' => $estudiante,
            'periodo' => [
                'fecha_inicio' => $fecha_inicio,
                'fecha_fin' => $fecha_fin
            ],
            'estadisticas' => [
                'total_dias' => $total_dias,
                'dias_asistidos' => $dias_asistidos,
                'total_asistencias' => $total_asistencias,
                'total_tardanzas' => $total_tardanzas,
                'total_faltas_justificadas' => $total_faltas_justificadas,
                'total_faltas_injustificadas' => $total_faltas_injustificadas,
                'total_minutos_tardanza' => $total_minutos_tardanza,
                'porcentaje_asistencia' => $porcentaje_asistencia,
                'porcentaje_faltas_injustificadas' => $porcentaje_faltas_injustificadas
            ],
            'alertas' => $alertas,
            'asistencias' => $asistencias
        ]
    ]);
}

/**
 * Detecta alertas basadas en las estadísticas del estudiante
 */
function detectar_alertas($porcentaje_asistencia, $porcentaje_faltas_injustificadas, $total_tardanzas, $total_faltas_injustificadas, $asistencias) {
    $alertas = [];
    $nivel_riesgo = 'bajo'; // bajo, medio, alto, critico

    // ALERTA 1: Riesgo de perder año (>30% faltas injustificadas)
    if ($porcentaje_faltas_injustificadas >= 30) {
        $alertas[] = [
            'tipo' => 'critico',
            'icono' => 'exclamation-triangle-fill',
            'titulo' => 'RIESGO DE PERDER AÑO',
            'mensaje' => "El estudiante tiene {$porcentaje_faltas_injustificadas}% de faltas injustificadas. Supera el límite del 30%.",
            'accion' => 'Intervención inmediata requerida. Contactar a apoderado y tutoría.',
            'codigo' => 'riesgo_perder_año'
        ];
        $nivel_riesgo = 'critico';
    } elseif ($porcentaje_faltas_injustificadas >= 20) {
        $alertas[] = [
            'tipo' => 'alto',
            'icono' => 'exclamation-circle-fill',
            'titulo' => 'ALERTA: Exceso de faltas injustificadas',
            'mensaje' => "El estudiante tiene {$porcentaje_faltas_injustificadas}% de faltas injustificadas. Se acerca al límite del 30%.",
            'accion' => 'Monitorear estrechamente. Considerar llamada a apoderado.',
            'codigo' => 'exceso_faltas'
        ];
        $nivel_riesgo = 'alto';
    }

    // ALERTA 2: Baja asistencia (<70%)
    if ($porcentaje_asistencia < 70) {
        $alertas[] = [
            'tipo' => 'alto',
            'icono' => 'emoji-frown-fill',
            'titulo' => 'Baja asistencia general',
            'mensaje' => "El estudiante solo asiste el {$porcentaje_asistencia}% de los días.",
            'accion' => 'Investigar causas. Revisar si necesita apoyo adicional.',
            'codigo' => 'baja_asistencia'
        ];
        if ($nivel_riesgo !== 'critico') {
            $nivel_riesgo = 'alto';
        }
    }

    // ALERTA 3: Exceso de tardanzas (más de 5 en el periodo)
    if ($total_tardanzas >= 10) {
        $alertas[] = [
            'tipo' => 'medio',
            'icono' => 'clock-fill',
            'titulo' => 'Exceso de tardanzas',
            'mensaje' => "El estudiante tiene {$total_tardanzas} tardanzas en el periodo.",
            'accion' => 'Considerar cambio de turno o revisar rutinas matutinas.',
            'codigo' => 'exceso_tardanzas'
        ];
        if ($nivel_riesgo === 'bajo') {
            $nivel_riesgo = 'medio';
        }
    } elseif ($total_tardanzas >= 5) {
        $alertas[] = [
            'tipo' => 'bajo',
            'icono' => 'clock-history',
            'titulo' => 'Tardanzas frecuentes',
            'mensaje' => "El estudiante tiene {$total_tardanzas} tardanzas en el periodo.",
            'accion' => 'Monitorear. Conversar con estudiante sobre punctualidad.',
            'codigo' => 'tardanzas_frecuentes'
        ];
    }

    // ALERTA 4: Faltas consecutivas (3 o más)
    $faltas_consecutivas = contar_faltas_consecutivas($asistencias);
    if ($faltas_consecutivas >= 3) {
        $alertas[] = [
            'tipo' => 'alto',
            'icono' => 'calendar-x-fill',
            'titulo' => 'Faltas consecutivas',
            'mensaje' => "El estudiante tiene {$faltas_consecutivas} faltas consecutivas.",
            'accion' => 'Contactar inmediatamente a la familia para verificar situación.',
            'codigo' => 'faltas_consecutivas'
        ];
        if ($nivel_riesgo !== 'critico' && $nivel_riesgo !== 'alto') {
            $nivel_riesgo = 'medio';
        }
    }

    // ALERTA 5: Patrón de ausencia (siempre falta el mismo día de la semana)
    $patron = detectar_patron_ausencia($asistencias);
    if ($patron) {
        $alertas[] = [
            'tipo' => 'medio',
            'icono' => 'calendar-week',
            'titulo' => 'Patrón de ausencia detectado',
            'mensaje' => "El estudiante falta frecuentemente los días {$patron['dia']}.",
            'accion' => "El {$porcentaje_asistencia}% de sus faltas son en {$patron['dia']}. Investigar causa.",
            'codigo' => 'patron_ausencia'
        ];
    }

    return $alertas;
}

/**
 * Cuenta faltas consecutivas recientes
 */
function contar_faltas_consecutivas($asistencias) {
    if (empty($asistencias)) {
        return 0;
    }

    $contador = 0;
    $hoy = date('Y-m-d');
    $ayer = date('Y-m-d', strtotime('-1 day'));

    // Ordenar ascendentemente por fecha
    usort($asistencias, function($a, $b) {
        return strtotime($a['fecha']) - strtotime($b['fecha']);
    });

    // Buscar faltas consecutivas empezando desde la fecha más reciente
    foreach (array_reverse($asistencias) as $a) {
        if (in_array($a['estado_codigo'], ['Falta Injustificada', 'Falta Justificada'])) {
            $contador++;
        } else {
            break;
        }
    }

    return $contador;
}

/**
 * Detecta si el estudiante falta siempre el mismo día de la semana
 */
function detectar_patron_ausencia($asistencias) {
    if (count($asistencias) < 5) {
        return null;
    }

    $dias_semana = [
        'Monday' => 'Lunes',
        'Tuesday' => 'Martes',
        'Wednesday' => 'Miércoles',
        'Thursday' => 'Jueves',
        'Friday' => 'Viernes'
    ];

    $faltas_por_dia = array_fill_keys(array_keys($dias_semana), 0);
    $total_faltas = 0;

    foreach ($asistencias as $a) {
        if (in_array($a['estado_codigo'], ['Falta Injustificada', 'Falta Justificada'])) {
            $dia_semana = date('l', strtotime($a['fecha']));
            if (isset($faltas_por_dia[$dia_semana])) {
                $faltas_por_dia[$dia_semana]++;
                $total_faltas++;
            }
        }
    }

    if ($total_faltas < 5) {
        return null;
    }

    // Buscar el día con más faltas
    arsort($faltas_por_dia);
    $dia_max = array_key_first($faltas_por_dia);
    $cantidad_max = $faltas_por_dia[$dia_max];
    $porcentaje = ($cantidad_max / $total_faltas) * 100;

    // Si más del 40% de las faltas son en un mismo día de la semana
    if ($porcentaje >= 40) {
        return [
            'dia' => $dias_semana[$dia_max],
            'cantidad' => $cantidad_max,
            'porcentaje' => round($porcentaje, 1)
        ];
    }

    return null;
}

/**
 * Obtiene estadísticas generales de asistencias (para dashboard)
 */
function obtener_estadisticas_generales() {
    $anio_lectivo_id = getAnioLectivoSeleccionado();
    $hoy = date('Y-m-d');

    // Estadísticas del día de hoy
    $stats_hoy = query("
        SELECT
            COUNT(*) as total_registros,
            SUM(CASE WHEN ea.conteo_asistencia = '1' THEN 1 ELSE 0 END) as asistencias,
            SUM(CASE WHEN ea.estado_codigo = 'Tardanza' OR ea.estado_codigo = 'Tardanza Justificada' THEN 1 ELSE 0 END) as tardanzas,
            SUM(CASE WHEN ea.estado_codigo = 'Falta Justificada' THEN 1 ELSE 0 END) as faltas_justificadas,
            SUM(CASE WHEN ea.estado_codigo = 'Falta Injustificada' THEN 1 ELSE 0 END) as faltas_injustificadas
        FROM asistencias a
        INNER JOIN estados_asistencia ea ON a.id_estado_asistencia = ea.id
        WHERE a.fecha = ?
          AND a.id_anio_lectivo = ?
    ", [$hoy, $anio_lectivo_id])->fetch();

    // Porcentaje de asistencia hoy
    $porcentaje_hoy = 0;
    if ($stats_hoy['total_registros'] > 0) {
        $porcentaje_hoy = round(($stats_hoy['asistencias'] / $stats_hoy['total_registros']) * 100, 1);
    }

    // Estadísticas del mes actual
    $stats_mes = query("
        SELECT
            COUNT(*) as total_registros,
            SUM(CASE WHEN ea.conteo_asistencia = '1' THEN 1 ELSE 0 END) as asistencias,
            SUM(CASE WHEN ea.estado_codigo = 'Tardanza' OR ea.estado_codigo = 'Tardanza Justificada' THEN 1 ELSE 0 END) as tardanzas,
            SUM(CASE WHEN ea.estado_codigo = 'Falta Justificada' THEN 1 ELSE 0 END) as faltas_justificadas,
            SUM(CASE WHEN ea.estado_codigo = 'Falta Injustificada' THEN 1 ELSE 0 END) as faltas_injustificadas
        FROM asistencias a
        INNER JOIN estados_asistencia ea ON a.id_estado_asistencia = ea.id
        WHERE YEAR(a.fecha) = YEAR(CURDATE())
          AND MONTH(a.fecha) = MONTH(CURDATE())
          AND a.id_anio_lectivo = ?
    ", [$anio_lectivo_id])->fetch();

    $porcentaje_mes = 0;
    if ($stats_mes['total_registros'] > 0) {
        $porcentaje_mes = round(($stats_mes['asistencias'] / $stats_mes['total_registros']) * 100, 1);
    }

    // Estadísticas del año lectivo
    $stats_anio = query("
        SELECT
            COUNT(*) as total_registros,
            SUM(CASE WHEN ea.conteo_asistencia = '1' THEN 1 ELSE 0 END) as asistencias,
            SUM(CASE WHEN ea.estado_codigo = 'Tardanza' OR ea.estado_codigo = 'Tardanza Justificada' THEN 1 ELSE 0 END) as tardanzas,
            SUM(CASE WHEN ea.estado_codigo = 'Falta Justificada' THEN 1 ELSE 0 END) as faltas_justificadas,
            SUM(CASE WHEN ea.estado_codigo = 'Falta Injustificada' THEN 1 ELSE 0 END) as faltas_injustificadas
        FROM asistencias a
        INNER JOIN estados_asistencia ea ON a.id_estado_asistencia = ea.id
        WHERE a.id_anio_lectivo = ?
    ", [$anio_lectivo_id])->fetch();

    $porcentaje_anio = 0;
    if ($stats_anio['total_registros'] > 0) {
        $porcentaje_anio = round(($stats_anio['asistencias'] / $stats_anio['total_registros']) * 100, 1);
    }

    // Total de estudiantes
    $total_estudiantes = query("
        SELECT COUNT(DISTINCT e.id) as total
        FROM estudiantes e
        INNER JOIN programaciones_estudiantes pe ON e.id = pe.id_estudiante
        WHERE pe.id_anio_lectivo = ?
          AND pe.estado = 'activo'
          AND e.estado = 'activo'
    ", [$anio_lectivo_id])->fetch()['total'];

    json_response([
        'success' => true,
        'data' => [
            'hoy' => array_merge($stats_hoy, ['porcentaje_asistencia' => $porcentaje_hoy]),
            'mes' => array_merge($stats_mes, ['porcentaje_asistencia' => $porcentaje_mes]),
            'anio' => array_merge($stats_anio, ['porcentaje_asistencia' => $porcentaje_anio]),
            'total_estudiantes' => $total_estudiantes
        ]
    ]);
}

/**
 * Obtiene alertas específicas para un estudiante
 */
function obtener_alertas_estudiante() {
    $id_estudiante = $_GET['id_estudiante'] ?? null;
    $anio_lectivo_id = getAnioLectivoSeleccionado();

    if (!$id_estudiante) {
        json_response(['success' => false, 'mensaje' => 'Falta el ID del estudiante'], 400);
    }

    // Reutilizar la lógica de obtener_reporte_individual
    $_GET['fecha_inicio'] = null;
    $_GET['fecha_fin'] = null;

    // Llamar a la función principal
    obtener_reporte_individual();
}

/**
 * Busca secciones/programaciones por nivel y turno
 */
function buscar_secciones() {
    $nivel_id = $_GET['nivel'] ?? null;
    $turno_id = $_GET['turno'] ?? null;
    $anio_lectivo_id = $_GET['anio_lectivo'] ?? getAnioLectivoSeleccionado();

    if (!$nivel_id || !$turno_id) {
        json_response([
            'success' => false,
            'mensaje' => 'Faltan parámetros: nivel y turno son requeridos'
        ], 400);
    }

    $secciones_result = query("
        SELECT
            p.id,
            p.id_anio_lectivo,
            n.nombre as nivel,
            g.grado,
            s.seccion,
            t.nombre as turno,
            t.abreviatura as turno_abreviatura,
            CONCAT(d.apellido_paterno, ' ', d.apellido_materno, ', ', d.nombres) as tutor_nombre,
            CONCAT(a.apellido_paterno, ' ', a.apellido_materno, ', ', a.nombres) as auxiliar_nombre
        FROM programaciones p
        INNER JOIN secciones s ON p.id_seccion = s.id
        INNER JOIN grados g ON s.id_grado = g.id
        INNER JOIN niveles n ON g.id_nivel = n.id
        INNER JOIN turnos t ON p.id_turno = t.id
        LEFT JOIN docentes d ON p.id_tutor = d.id
        LEFT JOIN auxiliares a ON p.id_auxiliar = a.id
        WHERE p.id_anio_lectivo = ?
          AND g.id_nivel = ?
          AND p.id_turno = ?
          AND p.estado = 'activo'
        ORDER BY n.nombre, g.grado, s.seccion
    ", [$anio_lectivo_id, $nivel_id, $turno_id]);

    if (!$secciones_result) {
        json_response([
            'success' => false,
            'mensaje' => 'Error al buscar secciones'
        ], 500);
    }

    $secciones = $secciones_result->fetchAll();

    json_response([
        'success' => true,
        'secciones' => $secciones,
        'total' => count($secciones)
    ]);
}

/**
 * Obtiene el reporte de asistencias de una sección/programación
 */
function obtener_reporte_seccion() {
    $programacion_id = $_GET['programacion_id'] ?? null;
    $fecha_inicio = $_GET['fecha_inicio'] ?? null;
    $fecha_fin = $_GET['fecha_fin'] ?? null;
    $anio_lectivo_id = getAnioLectivoSeleccionado();

    if (!$programacion_id) {
        json_response(['success' => false, 'mensaje' => 'Falta el ID de la programación'], 400);
    }

    // Si no se proporcionan fechas, usar todo el año lectivo
    if (!$fecha_inicio || !$fecha_fin) {
        $anio_result = query("SELECT fecha_inicio, fecha_fin FROM anios_lectivos WHERE id = ?", [$anio_lectivo_id]);
        if ($anio_result) {
            $anio_info = $anio_result->fetch();
            $fecha_inicio = $anio_info['fecha_inicio'];
            $fecha_fin = $anio_info['fecha_fin'];
        } else {
            $fecha_inicio = date('Y-m-01');
            $fecha_fin = date('Y-m-t');
        }
    }

    // Obtener información de la programación
    $prog_result = query("
        SELECT
            p.id,
            p.id_anio_lectivo,
            n.nombre as nivel,
            g.grado,
            s.seccion,
            t.nombre as turno,
            t.abreviatura as turno_abreviatura,
            CONCAT(doc_tutor.apellido_paterno, ' ', doc_tutor.apellido_materno, ', ', doc_tutor.nombres) as tutor_nombre,
            CONCAT(doc_auxiliar.apellido_paterno, ' ', doc_auxiliar.apellido_materno, ', ', doc_auxiliar.nombres) as auxiliar_nombre
        FROM programaciones p
        INNER JOIN secciones s ON p.id_seccion = s.id
        INNER JOIN grados g ON s.id_grado = g.id
        INNER JOIN niveles n ON g.id_nivel = n.id
        INNER JOIN turnos t ON p.id_turno = t.id
        LEFT JOIN docentes doc_tutor ON p.id_tutor = doc_tutor.id
        LEFT JOIN auxiliares doc_auxiliar ON p.id_auxiliar = doc_auxiliar.id
        WHERE p.id = ?
          AND p.estado = 'activo'
    ", [$programacion_id]);

    if (!$prog_result) {
        json_response(['success' => false, 'mensaje' => 'Programación no encontrada'], 404);
    }

    $programacion = $prog_result->fetch();

    if (!$programacion) {
        json_response(['success' => false, 'mensaje' => 'No se encontró la programación'], 404);
    }

    // Obtener estudiantes de la programación con sus estadísticas
    $estudiantes_result = query("
        SELECT
            e.id,
            e.dni,
            e.apellido_paterno,
            e.apellido_materno,
            e.nombres,
            COUNT(a.id) as total_registros
        FROM estudiantes e
        INNER JOIN programaciones_estudiantes pe ON e.id = pe.id_estudiante
        LEFT JOIN asistencias a ON e.id = a.id_estudiante
            AND a.id_anio_lectivo = ?
            AND a.fecha BETWEEN ? AND ?
        WHERE pe.id_programacion = ?
          AND pe.estado = 'activo'
          AND e.estado = 'activo'
        GROUP BY e.id, e.dni, e.apellido_paterno, e.apellido_materno, e.nombres
        ORDER BY e.apellido_paterno, e.apellido_materno, e.nombres
    ", [$anio_lectivo_id, $fecha_inicio, $fecha_fin, $programacion_id]);

    if (!$estudiantes_result) {
        json_response(['success' => false, 'mensaje' => 'Error al consultar estudiantes'], 500);
    }

    $estudiantes = $estudiantes_result->fetchAll();

    // Para cada estudiante, calcular sus estadísticas
    foreach ($estudiantes as &$est) {
        $stats_result = query("
            SELECT
                COUNT(*) as total_dias,
                SUM(CASE WHEN ea.conteo_asistencia = 'si' THEN 1 ELSE 0 END) as dias_asistidos,
                SUM(CASE WHEN a.estado_codigo = 'ASI' THEN 1 ELSE 0 END) as total_asistencias,
                SUM(CASE WHEN a.estado_codigo IN ('TAR', 'TJU') THEN 1 ELSE 0 END) as total_tardanzas,
                SUM(CASE WHEN a.estado_codigo = 'FJU' THEN 1 ELSE 0 END) as total_faltas_justificadas,
                SUM(CASE WHEN a.estado_codigo = 'FIN' THEN 1 ELSE 0 END) as total_faltas_injustificadas,
                SUM(a.minutos_tardanza) as total_minutos_tardanza
            FROM asistencias a
            INNER JOIN estados_asistencia ea ON a.id_estado_asistencia = ea.id
            WHERE a.id_estudiante = ?
              AND a.id_anio_lectivo = ?
              AND a.fecha BETWEEN ? AND ?
        ", [$est['id'], $anio_lectivo_id, $fecha_inicio, $fecha_fin])->fetch();

        $est = array_merge($est, $stats_result);

        // Calcular porcentaje
        $est['porcentaje_asistencia'] = $est['total_dias'] > 0
            ? round(($est['dias_asistidos'] / $est['total_dias']) * 100, 1)
            : 0;
    }

    // Calcular estadísticas generales de la sección
    $total_estudiantes = count($estudiantes);
    $total_asistencias = array_sum(array_column($estudiantes, 'total_asistencias'));
    $total_tardanzas = array_sum(array_column($estudiantes, 'total_tardanzas'));
    $total_faltas_justificadas = array_sum(array_column($estudiantes, 'total_faltas_justificadas'));
    $total_faltas_injustificadas = array_sum(array_column($estudiantes, 'total_faltas_injustificadas'));

    $total_dias_seccion = array_sum(array_column($estudiantes, 'total_dias'));
    $total_dias_asistidos = array_sum(array_column($estudiantes, 'dias_asistidos'));
    $porcentaje_asistencia = $total_dias_seccion > 0
        ? round(($total_dias_asistidos / $total_dias_seccion) * 100, 1)
        : 0;

    // Presentes hoy
    $hoy = date('Y-m-d');
    $presentes_hoy_result = query("
        SELECT COUNT(DISTINCT a.id_estudiante) as presentes
        FROM asistencias a
        INNER JOIN programaciones_estudiantes pe ON a.id_estudiante = pe.id_estudiante
        WHERE pe.id_programacion = ?
          AND a.fecha = ?
          AND a.id_anio_lectivo = ?
          AND pe.estado = 'activo'
    ", [$programacion_id, $hoy, $anio_lectivo_id])->fetch();

    $presentes_hoy = $presentes_hoy_result['presentes'] ?? 0;

    json_response([
        'success' => true,
        'data' => [
            'programacion' => $programacion,
            'estadisticas' => [
                'total_estudiantes' => $total_estudiantes,
                'presentes_hoy' => $presentes_hoy,
                'total_asistencias' => $total_asistencias,
                'total_tardanzas' => $total_tardanzas,
                'total_faltas_justificadas' => $total_faltas_justificadas,
                'total_faltas_injustificadas' => $total_faltas_injustificadas,
                'porcentaje_asistencia' => $porcentaje_asistencia
            ],
            'periodo' => [
                'fecha_inicio' => $fecha_inicio,
                'fecha_fin' => $fecha_fin
            ],
            'estudiantes' => $estudiantes
        ]
    ]);
}

/**
 * Obtiene la grilla de asistencias para una sección (formato PDF mensual)
 */
function obtener_grilla_seccion() {
    $programacion_id = $_GET['programacion_id'] ?? null;
    $fecha_inicio = $_GET['fecha_inicio'] ?? null;
    $fecha_fin = $_GET['fecha_fin'] ?? null;
    $anio_lectivo_id = getAnioLectivoSeleccionado();

    if (!$programacion_id) {
        json_response(['success' => false, 'mensaje' => 'Falta el ID de la programación'], 400);
    }

    // Si no se proporcionan fechas, usar el mes actual
    if (!$fecha_inicio || !$fecha_fin) {
        $fecha_inicio = date('Y-m-01');
        $fecha_fin = date('Y-m-t');
    }

    // Obtener información de la programación
    $prog_result = query("
        SELECT
            p.id,
            n.nombre as nivel,
            g.grado,
            s.seccion,
            t.nombre as turno
        FROM programaciones p
        INNER JOIN secciones s ON p.id_seccion = s.id
        INNER JOIN grados g ON s.id_grado = g.id
        INNER JOIN niveles n ON g.id_nivel = n.id
        INNER JOIN turnos t ON p.id_turno = t.id
        WHERE p.id = ?
          AND p.estado = 'activo'
    ", [$programacion_id]);

    if (!$prog_result) {
        json_response(['success' => false, 'mensaje' => 'Programación no encontrada'], 404);
    }

    $programacion = $prog_result->fetch();

    if (!$programacion) {
        json_response(['success' => false, 'mensaje' => 'No se encontró la programación'], 404);
    }

    // Obtener estudiantes de la programación
    $estudiantes_result = query("
        SELECT
            e.id,
            e.dni,
            e.apellido_paterno,
            e.apellido_materno,
            e.nombres
        FROM estudiantes e
        INNER JOIN programaciones_estudiantes pe ON e.id = pe.id_estudiante
        WHERE pe.id_programacion = ?
          AND pe.estado = 'activo'
          AND e.estado = 'activo'
        ORDER BY e.apellido_paterno, e.apellido_materno, e.nombres
    ", [$programacion_id]);

    if (!$estudiantes_result) {
        json_response(['success' => false, 'mensaje' => 'Error al consultar estudiantes'], 500);
    }

    $estudiantes = $estudiantes_result->fetchAll();

    // Generar array de fechas (solo días hábiles: lunes a viernes)
    $fechas = [];
    $fecha_actual = strtotime($fecha_inicio);
    $fecha_fin_timestamp = strtotime($fecha_fin);

    while ($fecha_actual <= $fecha_fin_timestamp) {
        $dia_semana = date('w', $fecha_actual); // 0=Domingo, 6=Sábado

        // Solo incluir lunes a viernes (1-5)
        if ($dia_semana >= 1 && $dia_semana <= 5) {
            $fechas[] = date('Y-m-d', $fecha_actual);
        }

        $fecha_actual = strtotime('+1 day', $fecha_actual);
    }

    // Para cada estudiante, obtener sus asistencias en las fechas
    $grilla = [];

    foreach ($estudiantes as $est) {
        $asistencias_result = query("
            SELECT
                a.fecha,
                a.estado_codigo
            FROM asistencias a
            WHERE a.id_estudiante = ?
              AND a.id_anio_lectivo = ?
              AND a.fecha IN ('" . implode("','", $fechas) . "')
        ", [$est['id'], $anio_lectivo_id]);

        if ($asistencias_result) {
            $asistencias = $asistencias_result->fetchAll();

            // Crear mapa de fecha -> estado_codigo
            $asistencias_por_fecha = [];
            foreach ($asistencias as $a) {
                $asistencias_por_fecha[$a['fecha']] = $a['estado_codigo'];
            }

            $grilla[$est['id']] = $asistencias_por_fecha;
        }
    }

    // Obtener parámetros de la institución
    $cfg_result = query("SELECT parametro, valor FROM configuracion WHERE parametro IN ('nombre_institucion', 'logo_institucion')");
    $config = [];
    while ($row = $cfg_result->fetch()) {
        $config[$row['parametro']] = $row['valor'];
    }

    json_response([
        'success' => true,
        'data' => [
            'institucion' => [
                'nombre' => $config['nombre_institucion'] ?? 'SICA',
                'logo' => $config['logo_institucion'] ?? ''
            ],
            'programacion' => $programacion,
            'periodo' => [
                'fecha_inicio' => $fecha_inicio,
                'fecha_fin' => $fecha_fin
            ],
            'estudiantes' => $estudiantes,
            'fechas' => $fechas,
            'grilla' => $grilla
        ]
    ]);
}
