<?php
require_once '../config/db.php';
require_once '../config/functions.php';
requerirAutenticacion();

$anio_seleccionado = getAnioLectivoSeleccionado();
$info_anio = getInfoAnioLectivoSeleccionado();

// Obtener niveles activos
$niveles = query("SELECT id, nombre FROM niveles WHERE estado = 'activo' ORDER BY nombre")->fetchAll();

// Obtener docentes activos
$docentes = query("
    SELECT d.id, d.dni,
           CONCAT(d.apellido_paterno, ' ', d.apellido_materno, ', ', d.nombres) AS nombre
    FROM docentes d
    WHERE d.estado = 'activo'
    ORDER BY d.apellido_paterno, d.apellido_materno
")->fetchAll();

// Obtener auxiliares activos
$auxiliares = query("
    SELECT a.id, a.dni,
           CONCAT(a.apellido_paterno, ' ', a.apellido_materno, ', ', a.nombres) AS nombre
    FROM auxiliares a
    WHERE a.estado = 'activo'
    ORDER BY a.apellido_paterno, a.apellido_materno
")->fetchAll();

$errores = [];
$exito = false;

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $id_seccion = $_POST['id_seccion'] ?? 0;
    $id_tutor = $_POST['id_tutor'] ?? null;
    $id_auxiliar = $_POST['id_auxiliar'] ?? null;
    $observaciones = $_POST['observaciones'] ?? null;

    // Validaciones
    if (!$id_seccion) {
        $errores[] = "Debe seleccionar una sección";
    }

    if (empty($errores)) {
        try {
            $pdo->beginTransaction();

            // Verificar que no exista programación para esta sección
            $stmt = $pdo->prepare("
                SELECT COUNT(*)
                FROM programaciones
                WHERE id_seccion = ? AND id_anio_lectivo = ? AND estado != 'inactivo'
            ");
            $stmt->execute([$id_seccion, $anio_seleccionado]);

            if ($stmt->fetchColumn() > 0) {
                throw new Exception("Ya existe una programación activa para esta sección en el año lectivo actual");
            }

            // Obtener turno y capacidad de la sección
            $stmt = $pdo->prepare("
                SELECT id_turno, capacidad
                FROM secciones
                WHERE id = ?
            ");
            $stmt->execute([$id_seccion]);
            $seccion_info = $stmt->fetch();

            if (!$seccion_info) {
                throw new Exception("La sección seleccionada no existe");
            }

            // Insertar programación con datos de la sección
            $stmt = $pdo->prepare("
                INSERT INTO programaciones
                (id_seccion, id_anio_lectivo, id_tutor, id_auxiliar, id_turno, capacidad, observaciones, estado)
                VALUES (?, ?, ?, ?, ?, ?, ?, 'activo')
            ");

            $stmt->execute([
                $id_seccion,
                $anio_seleccionado,
                $id_tutor ?: null,
                $id_auxiliar ?: null,
                $seccion_info['id_turno'],
                $seccion_info['capacidad'],
                $observaciones
            ]);

            $id_programacion = $pdo->lastInsertId();

            $pdo->commit();
            $exito = true;

            // Redirigir a ver.php o a asignar estudiantes
            header("Location: ver.php?id=$id_programacion&exito=1");
            exit;

        } catch (Exception $e) {
            $pdo->rollBack();
            $errores[] = $e->getMessage();
        }
    }
}
?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0, maximum-scale=1.0, user-scalable=no">
    <title>Nueva Programación - SICA</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.0/font/bootstrap-icons.css" rel="stylesheet">
    <link href="../assets/css/sica.css?v=5.1" rel="stylesheet">
</head>
<body>
    <?php include '../includes/layout_elems.php'; ?>

    
        <div class="d-flex justify-content-between align-items-center mb-4">
            <div class="d-flex align-items-center">
                <button class="btn btn-link d-lg-none me-3" onclick="toggleSidebarMobile()">
                    <i class="bi bi-list fs-4"></i>
                </button>
                <div>
                    <h3 class="mb-1">Nueva Programación</h3>
                    <p class="text-muted mb-0">Crear una nueva programación académica</p>
                </div>
            </div>
            <a href="index.php" class="btn btn-outline-secondary">
                <i class="bi bi-arrow-left me-2"></i>Volver
            </a>
        </div>

        <?php if (!empty($errores)): ?>
            <div class="alert alert-danger">
                <i class="bi bi-exclamation-triangle me-2"></i>
                <strong>Error:</strong>
                <ul class="mb-0 mt-2">
                    <?php foreach ($errores as $error): ?>
                        <li><?= htmlspecialchars($error) ?></li>
                    <?php endforeach; ?>
                </ul>
            </div>
        <?php endif; ?>

        <div class="row">
            <div class="col-lg-8">
                <div class="card border-0 shadow-sm">
                    <div class="card-body p-4">
                        <form method="POST" id="formCrear">
                            <!-- Información del Año Lectivo -->
                            <div class="alert alert-info mb-4">
                                <i class="bi bi-info-circle me-2"></i>
                                Año Lectivo: <strong><?= htmlspecialchars($info_anio['anio']) ?></strong>
                            </div>

                            <!-- Paso 1: Sección -->
                            <div class="mb-4">
                                <h5 class="mb-3">
                                    <span class="badge bg-primary">1</span>
                                    Seleccionar Sección
                                </h5>

                                <div class="row">
                                    <div class="col-md-6 mb-3">
                                        <label class="form-label fw-semibold">Nivel <span class="text-danger">*</span></label>
                                        <select name="id_nivel" class="form-select" id="id_nivel" required>
                                            <option value="">Seleccionar nivel...</option>
                                            <?php foreach ($niveles as $nivel): ?>
                                                <option value="<?= $nivel['id'] ?>">
                                                    <?= htmlspecialchars($nivel['nombre']) ?>
                                                </option>
                                            <?php endforeach; ?>
                                        </select>
                                    </div>
                                    <div class="col-md-6 mb-3">
                                        <label class="form-label fw-semibold">Grado y Sección <span class="text-danger">*</span></label>
                                        <select name="id_seccion" class="form-select" id="id_seccion" required disabled>
                                            <option value="">Primero selecciona nivel...</option>
                                        </select>
                                        <small class="text-muted">
                                            Solo se muestran secciones sin programación activa
                                        </small>
                                        <div id="info-seccion" class="mt-2" style="display: none;">
                                            <small class="text-info">
                                                <i class="bi bi-info-circle me-1"></i>
                                                <span id="detalles-seccion"></span>
                                            </small>
                                        </div>
                                    </div>
                                </div>
                            </div>

                            <!-- Paso 2: Tutor -->
                            <div class="mb-4">
                                <h5 class="mb-3">
                                    <span class="badge bg-primary">2</span>
                                    Tutor de la Sección
                                </h5>

                                <div class="mb-3">
                                    <label class="form-label fw-semibold">Docente Tutor</label>
                                    <select name="id_tutor" class="form-select">
                                        <option value="">Sin tutor (asignar después)</option>
                                        <?php foreach ($docentes as $doc): ?>
                                            <option value="<?= $doc['id'] ?>">
                                                <?= htmlspecialchars($doc['nombre']) ?> (DNI: <?= $doc['dni'] ?>)
                                            </option>
                                        <?php endforeach; ?>
                                    </select>
                                    <small class="text-muted">
                                        El tutor será el docente principal de la sección
                                    </small>
                                </div>
                            </div>

                            <!-- Paso 3: Auxiliares -->
                            <div class="mb-4">
                                <h5 class="mb-3">
                                    <span class="badge bg-primary">3</span>
                                    Auxiliar de Educación
                                </h5>

                                <div class="mb-3">
                                    <label class="form-label fw-semibold">Auxiliar Asignado</label>
                                    <select name="id_auxiliar" class="form-select">
                                        <option value="">Sin auxiliar asignado</option>
                                        <?php foreach ($auxiliares as $aux): ?>
                                            <option value="<?= $aux['id'] ?>">
                                                <?= htmlspecialchars($aux['nombre']) ?>
                                                <small>(DNI: <?= $aux['dni'] ?>)</small>
                                            </option>
                                        <?php endforeach; ?>
                                    </select>
                                    <small class="text-muted">
                                        Cada sección tiene un solo auxiliar asignado
                                    </small>
                                </div>
                            </div>

                            <!-- Observaciones -->
                            <div class="mb-4">
                                <label class="form-label fw-semibold">Observaciones</label>
                                <textarea name="observaciones" class="form-control" rows="3"
                                          placeholder="Comentarios adicionales sobre la programación..."></textarea>
                            </div>

                            <!-- Botones -->
                            <div class="d-flex gap-2">
                                <button type="submit" class="btn btn-primary btn-lg">
                                    <i class="bi bi-check-circle me-2"></i>Crear Programación
                                </button>
                                <a href="index.php" class="btn btn-outline-secondary btn-lg">
                                    <i class="bi bi-x-circle me-2"></i>Cancelar
                                </a>
                            </div>
                        </form>
                    </div>
                </div>
            </div>

            <!-- Sidebar derecho con información -->
            <div class="col-lg-4">
                <div class="card border-0 shadow-sm mb-4">
                    <div class="card-header bg-primary text-white">
                        <h6 class="mb-0"><i class="bi bi-info-circle me-2"></i>Instrucciones</h6>
                    </div>
                    <div class="card-body">
                        <ol class="mb-0">
                            <li class="mb-2">Selecciona el <strong>nivel educativo</strong></li>
                            <li class="mb-2">Elige la <strong>sección</strong> (grado y letra)</li>
                            <li class="mb-2">Asigna un <strong>tutor</strong> (opcional)</li>
                            <li class="mb-2">Selecciona el <strong>auxiliar</strong></li>
                            <li class="mb-0">Luego podrás asignar estudiantes y áreas académicas</li>
                        </ol>
                    </div>
                </div>

                <div class="card border-0 shadow-sm">
                    <div class="card-header bg-success text-white">
                        <h6 class="mb-0"><i class="bi bi-lightbulb me-2"></i>Próximos Pasos</h6>
                    </div>
                    <div class="card-body">
                        <ul class="mb-0">
                            <li class="mb-2"><i class="bi bi-check2 text-success me-2"></i>Asignar estudiantes</li>
                            <li class="mb-2"><i class="bi bi-check2 text-success me-2"></i>Asignar áreas académicas</li>
                            <li class="mb-2"><i class="bi bi-check2 text-success me-2"></i>Asignar docentes a áreas</li>
                            <li class="mb-0"><i class="bi bi-check2 text-success me-2"></i>Generar reportes</li>
                        </ul>
                    </div>
                </div>
            </div>
        </div>
    </div>

    </div>
    <?php include '../includes/layout_scripts.php'; ?>

    <script>
    const idNivelSelect = document.getElementById('id_nivel');
    const idSeccionSelect = document.getElementById('id_seccion');

    idNivelSelect.addEventListener('change', function() {
        const idNivel = this.value;

        if (!idNivel) {
            idSeccionSelect.disabled = true;
            idSeccionSelect.innerHTML = '<option value="">Primero selecciona nivel...</option>';
            document.getElementById('info-seccion').style.display = 'none';
            return;
        }

        // Cargar secciones disponibles mediante AJAX
        fetch('ajax.php', {
            method: 'POST',
            headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
            body: `accion=get_secciones&id_nivel=${idNivel}&id_anio_lectivo=<?= $anio_seleccionado ?>`
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                if (data.secciones.length === 0) {
                    idSeccionSelect.innerHTML = '<option value="">No hay secciones disponibles</option>';
                } else {
                    idSeccionSelect.innerHTML = '<option value="">Seleccionar sección...</option>';
                    data.secciones.forEach(seccion => {
                        const option = document.createElement('option');
                        option.value = seccion.id;
                        option.textContent = `${seccion.grado}° - Sección ${seccion.seccion}`;
                        option.dataset.turnoNombre = seccion.turno;
                        option.dataset.turnoAbrev = seccion.turno_abrev;
                        option.dataset.horaIngreso = seccion.hora_ingreso;
                        option.dataset.horaSalida = seccion.hora_salida;
                        option.dataset.capacidad = seccion.capacidad;
                        idSeccionSelect.appendChild(option);
                    });
                }
                idSeccionSelect.disabled = false;
            } else {
                alert('Error: ' + data.mensaje);
            }
        })
        .catch(error => {
            console.error('Error:', error);
            alert('Error al cargar secciones');
        });
    });

    // Mostrar detalles de la sección seleccionada
    idSeccionSelect.addEventListener('change', function() {
        const selectedOption = this.options[this.selectedIndex];
        const infoDiv = document.getElementById('info-seccion');
        const detallesSpan = document.getElementById('detalles-seccion');

        if (this.value && selectedOption.dataset.turnoNombre) {
            const turno = selectedOption.dataset.turnoNombre;
            const horaIngreso = selectedOption.dataset.horaIngreso ? selectedOption.dataset.horaIngreso.substring(0, 5) : '--:--';
            const horaSalida = selectedOption.dataset.horaSalida ? selectedOption.dataset.horaSalida.substring(0, 5) : '--:--';
            detallesSpan.textContent = `Turno: ${turno} (${horaIngreso} - ${horaSalida}) | Capacidad: ${selectedOption.dataset.capacidad} estudiantes`;
            infoDiv.style.display = 'block';
        } else {
            infoDiv.style.display = 'none';
        }
    });

    // Mantener selección si hay error
    <?php if (isset($_POST['id_nivel']) && isset($_POST['id_seccion'])): ?>
        document.addEventListener('DOMContentLoaded', function() {
            idNivelSelect.value = '<?= $_POST['id_nivel'] ?>';
            idNivelSelect.dispatchEvent(new Event('change'));

            setTimeout(() => {
                idSeccionSelect.value = '<?= $_POST['id_seccion'] ?>';
            }, 500);
        });
    <?php endif; ?>
    </script>
</body>
</html>
