<?php
require_once '../config/db.php';
require_once '../config/functions.php';
requerirAutenticacion();

$id = $_GET['id'] ?? 0;

if (!$id) {
    header('Location: index.php');
    exit;
}

// Obtener programación
$stmt = $pdo->prepare("
    SELECT
        p.id,
        p.id_seccion,
        p.id_anio_lectivo,
        p.id_tutor,
        p.id_auxiliar,
        p.capacidad,
        p.estado,
        p.observaciones,
        t.nombre as turno,
        t.abreviatura as turno_abrev,
        t.hora_ingreso,
        t.hora_salida,
        s.id_nivel,
        n.nombre AS nivel_nombre,
        s.grado,
        s.seccion,
        a.anio
    FROM programaciones p
    INNER JOIN secciones s ON s.id = p.id_seccion
    INNER JOIN turnos t ON t.id = p.id_turno
    INNER JOIN niveles n ON n.id = s.id_nivel
    INNER JOIN anios_lectivos a ON a.id = p.id_anio_lectivo
    WHERE p.id = ?
");
$stmt->execute([$id]);
$programacion = $stmt->fetch();

if (!$programacion) {
    header('Location: index.php');
    exit;
}

// Obtener áreas ya asignadas a esta programación
$stmt = $pdo->prepare("
    SELECT
        pda.id,
        pda.id_area_academica,
        pda.id_docente,
        aa.area,
        CONCAT(d.apellido_paterno, ' ', d.apellido_materno, ', ', d.nombres) AS docente_nombre,
        pda.fecha_asignacion
    FROM programaciones_docentes_areas pda
    INNER JOIN areas_academicas aa ON aa.id = pda.id_area_academica
    INNER JOIN docentes d ON d.id = pda.id_docente
    WHERE pda.id_programacion = ? AND pda.estado = 'activo'
    ORDER BY d.apellido_paterno, d.apellido_materno, aa.area
");
$stmt->execute([$id]);
$areas_asignadas = $stmt->fetchAll();

// Asignar áreas (docente + múltiples áreas seleccionadas)
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['id_docente']) && isset($_POST['areas'])) {
    $id_docente = $_POST['id_docente'];
    $areas_seleccionadas = $_POST['areas']; // Array de IDs

    try {
        if (empty($areas_seleccionadas)) {
            throw new Exception('Debe seleccionar al menos un área.');
        }

        $asignadas = 0;
        $duplicadas = 0;

        foreach ($areas_seleccionadas as $id_area) {
            // Verificar duplicados (incluyendo inactivos por el índice único)
            $stmt = $pdo->prepare("
                SELECT COUNT(*) as c
                FROM programaciones_docentes_areas
                WHERE id_programacion = ?
                AND id_docente = ?
                AND id_area_academica = ?
            ");
            $stmt->execute([$id, $id_docente, $id_area]);
            $existe = $stmt->fetch()['c'];

            if ($existe > 0) {
                // Ya existe (activa o inactiva), verificar si está inactiva para reactivar
                $stmt = $pdo->prepare("
                    SELECT estado, id
                    FROM programaciones_docentes_areas
                    WHERE id_programacion = ?
                    AND id_docente = ?
                    AND id_area_academica = ?
                    LIMIT 1
                ");
                $stmt->execute([$id, $id_docente, $id_area]);
                $registro = $stmt->fetch();

                if ($registro['estado'] === 'inactivo') {
                    // Reactivar en lugar de insertar nuevo
                    $stmt = $pdo->prepare("
                        UPDATE programaciones_docentes_areas
                        SET estado = 'activo', fecha_asignacion = CURDATE()
                        WHERE id = ?
                    ");
                    $stmt->execute([$registro['id']]);
                    $asignadas++;
                } else {
                    $duplicadas++;
                }
                continue;
            }

            // Insertar nueva asignación
            try {
                $stmt = $pdo->prepare("
                    INSERT INTO programaciones_docentes_areas
                    (id_programacion, id_docente, id_area_academica, fecha_asignacion, estado)
                    VALUES (?, ?, ?, CURDATE(), 'activo')
                ");
                $stmt->execute([$id, $id_docente, $id_area]);
                $asignadas++;
            } catch (PDOException $e) {
                // Si es error de duplicado (código 23000), incrementar duplicadas
                if ($e->getCode() == '23000') {
                    $duplicadas++;
                } else {
                    throw $e; // Re-lanzar otros errores
                }
            }
        }

        if ($asignadas == 0 && $duplicadas > 0) {
            throw new Exception('Todas las áreas seleccionadas ya estaban asignadas.');
        } elseif ($asignadas == 0) {
            throw new Exception('No se pudo asignar ninguna área.');
        }

        $mensaje = $asignadas . ' área(s) asignada(s) correctamente';
        if ($duplicadas > 0) {
            $mensaje .= '. ' . $duplicadas . ' área(s) ya estaban asignadas y se omitieron.';
        }

        header('Location: asignar_docentes.php?id=' . $id . '&exito=' . urlencode($mensaje));
        exit;

    } catch (PDOException $e) {
        // Capturar error de duplicado específico
        if ($e->getCode() == '23000') {
            $error = 'Una de las áreas seleccionadas ya está asignada a este docente en esta programación.';
        } else {
            $error = $e->getMessage();
        }
    } catch (Exception $e) {
        $error = $e->getMessage();
    }
}

// Desasignar área
if (isset($_GET['desasignar'])) {
    $id_asignacion = $_GET['desasignar'];

    try {
        $stmt = $pdo->prepare("
            UPDATE programaciones_docentes_areas
            SET estado = 'inactivo'
            WHERE id = ? AND id_programacion = ?
        ");
        $stmt->execute([$id_asignacion, $id]);

        header("Location: asignar_docentes.php?id=$id&exito_desasignacion=1");
        exit;
    } catch (Exception $e) {
        $error = $e->getMessage();
    }
}

// Obtener docentes disponibles
$docentes = query("
    SELECT d.id, d.dni,
           CONCAT(d.apellido_paterno, ' ', d.apellido_materno, ', ', d.nombres) AS nombre
    FROM docentes d
    WHERE d.estado = 'activo'
    ORDER BY d.apellido_paterno, d.apellido_materno
")->fetchAll();
?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0, maximum-scale=1.0, user-scalable=no">
    <title>Asignar Áreas - SICA</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.0/font/bootstrap-icons.css" rel="stylesheet">
    <link href="../assets/css/sica.css?v=5.1" rel="stylesheet">
    <style>
        .area-checkbox {
            display: none;
        }
        .area-card {
            cursor: pointer;
            transition: all 0.2s;
            border: 2px solid transparent;
        }
        .area-card:hover {
            transform: translateY(-2px);
            box-shadow: 0 4px 12px rgba(0,0,0,0.15);
        }
        .area-checkbox:checked + .area-card {
            border-color: #0d6efd;
            background-color: #e7f1ff;
        }
        .area-checkbox:checked + .area-card .card-body {
            color: #0d6efd;
        }
    </style>
</head>
<body>
    <?php include '../includes/layout_elems.php'; ?>

    <div class="d-flex justify-content-between align-items-center mb-4">
        <div class="d-flex align-items-center">
            <button class="btn btn-link d-lg-none me-3" onclick="toggleSidebarMobile()">
                <i class="bi bi-list fs-4"></i>
            </button>
            <div>
                <nav aria-label="breadcrumb">
                    <ol class="breadcrumb mb-1">
                        <li class="breadcrumb-item"><a href="index.php" class="text-decoration-none">Programaciones</a></li>
                        <li class="breadcrumb-item"><a href="ver.php?id=<?= $id ?>" class="text-decoration-none">Ver</a></li>
                        <li class="breadcrumb-item active">Asignar Áreas</li>
                    </ol>
                </nav>
                <h3 class="mb-0">Asignar Áreas Académicas</h3>
            </div>
        </div>
        <a href="ver.php?id=<?= $id ?>" class="btn btn-outline-secondary">
            <i class="bi bi-arrow-left me-2"></i>Volver
        </a>
    </div>

    <?php if (isset($error)): ?>
        <div class="alert alert-danger">
            <i class="bi bi-exclamation-triangle me-2"></i><?= htmlspecialchars($error) ?>
        </div>
    <?php endif; ?>

    <?php if (isset($_GET['exito'])): ?>
        <div class="alert alert-success">
            <i class="bi bi-check-circle me-2"></i><?= htmlspecialchars(urldecode($_GET['exito'])) ?>
        </div>
    <?php endif; ?>

    <?php if (isset($_GET['exito_desasignacion'])): ?>
        <div class="alert alert-success">
            <i class="bi bi-check-circle me-2"></i>Área desasignada correctamente.
        </div>
    <?php endif; ?>

    <!-- Información de la Programación -->
    <div class="card border-0 shadow-sm mb-4">
        <div class="card-body">
            <h5 class="mb-3">
                <?= htmlspecialchars($programacion['nivel_nombre']) ?> -
                <?= htmlspecialchars($programacion['grado']) ?>°
                <strong><?= htmlspecialchars($programacion['seccion']) ?></strong>
                <span class="badge bg-primary ms-2"><?= count($areas_asignadas) ?> áreas</span>
            </h5>
        </div>
    </div>

    <div class="row">
        <!-- Columna Izquierda: Formulario de Asignación -->
        <div class="col-lg-5">
            <div class="card border-0 shadow-sm mb-4">
                <div class="card-header bg-primary text-white">
                    <h6 class="mb-0"><i class="bi bi-plus-circle me-2"></i>Asignar Nueva Área</h6>
                </div>
                <div class="card-body">
                    <form method="POST" id="formAsignar">
                        <div class="mb-3">
                            <label class="form-label fw-semibold">Docente <span class="text-danger">*</span></label>
                            <select name="id_docente" class="form-select" required id="id_docente">
                                <option value="">Seleccionar docente...</option>
                                <?php foreach ($docentes as $doc): ?>
                                    <option value="<?= $doc['id'] ?>">
                                        <?= htmlspecialchars($doc['nombre']) ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                            <small class="form-text text-muted">
                               Seleccione el docente que impartirá las áreas
                            </small>
                        </div>

                        <div class="mb-3" id="contenedor_areas">
                            <label class="form-label fw-semibold">
                                Áreas Académicas (<?= htmlspecialchars($programacion['nivel_nombre']) ?>) <span class="text-danger">*</span>
                            </label>
                            <small class="text-muted d-block mb-2">Seleccione una o más áreas para asignar (nivel: <?= htmlspecialchars($programacion['nivel_nombre']) ?>)</small>
                            <div id="areas_disponibles" class="row g-2">
                                <!-- Las áreas se cargarán aquí con AJAX -->
                            </div>
                        </div>

                        <button type="submit" class="btn btn-primary w-100" id="btnAsignar" disabled>
                            <i class="bi bi-check-circle me-2"></i>Asignar Áreas Seleccionadas
                        </button>
                    </form>
                </div>
            </div>
        </div>

        <!-- Columna Derecha: Áreas Asignadas -->
        <div class="col-lg-7">
            <div class="card border-0 shadow-sm">
                <div class="card-header bg-success text-white">
                    <h6 class="mb-0">
                        <i class="bi bi-book me-2"></i>Áreas Asignadas
                        <span class="badge bg-light text-dark ms-2"><?= count($areas_asignadas) ?></span>
                    </h6>
                </div>
                <div class="card-body">
                    <?php if (empty($areas_asignadas)): ?>
                        <div class="text-center py-5">
                            <i class="bi bi-book-x fs-1 text-muted"></i>
                            <p class="text-muted mt-3">No hay áreas asignadas</p>
                        </div>
                    <?php else: ?>
                        <div class="table-responsive">
                            <table class="table table-hover">
                                <thead class="table-light">
                                    <tr>
                                        <th>Docente</th>
                                        <th>Área</th>
                                        <th>Fecha Asignación</th>
                                        <th>Acciones</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($areas_asignadas as $area): ?>
                                        <tr>
                                            <td>
                                                <strong><?= htmlspecialchars($area['docente_nombre']) ?></strong>
                                            </td>
                                            <td><?= htmlspecialchars($area['area']) ?></td>
                                            <td>
                                                <small class="text-muted">
                                                    <?= date('d/m/Y', strtotime($area['fecha_asignacion'])) ?>
                                                </small>
                                            </td>
                                            <td>
                                                <a href="?id=<?= $id ?>&desasignar=<?= $area['id'] ?>"
                                                   class="btn btn-sm btn-outline-danger"
                                                   onclick="return confirm('¿Desasignar área de esta programación?')">
                                                    <i class="bi bi-trash"></i>
                                                </a>
                                            </td>
                                        </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                    <?php endif; ?>
                </div>
            </div>
        </div>
    </div>

    <?php include '../includes/layout_scripts.php'; ?>
    <script>
        // Cargar áreas automáticamente al cargar la página
        cargarAreasPorNivel();

        function cargarAreasPorNivel() {
            const contenedor = document.getElementById('contenedor_areas');
            const areasDiv = document.getElementById('areas_disponibles');
            const btnAsignar = document.getElementById('btnAsignar');

            // Cargar áreas con AJAX según el nivel de la programación
            const url = 'ajax.php?accion=get_areas_docente&id_anio_lectivo=<?= $programacion['id_anio_lectivo'] ?>&id_programacion=<?= $id ?>';

            fetch(url)
                .then(response => {
                    if (!response.ok) {
                        throw new Error('Error en la respuesta: ' + response.status);
                    }
                    return response.json();
                })
                .then(data => {
                    if (!data.success) {
                        throw new Error(data.mensaje || 'Error desconocido');
                    }

                    if (data.areas && data.areas.length > 0) {
                        areasDiv.innerHTML = data.areas.map(area => `
                            <div class="col-6">
                                <label class="mb-0">
                                    <input type="checkbox" name="areas[]" value="${area.id}" class="area-checkbox" onchange="verificarSeleccion()">
                                    <div class="card area-card">
                                        <div class="card-body p-2">
                                            <small class="fw-semibold">${area.area}</small>
                                            <br>
                                            <small class="text-muted" style="font-size: 0.7rem;">${area.nivel_nombre}</small>
                                        </div>
                                    </div>
                                </label>
                            </div>
                        `).join('');
                        contenedor.style.display = 'block';
                        btnAsignar.disabled = true;
                    } else {
                        areasDiv.innerHTML = '<div class="col-12"><p class="text-muted">No hay áreas académicas configuradas para el nivel ' + '<?= htmlspecialchars($programacion['nivel_nombre']) ?>' + ' en el año lectivo actual.</p></div>';
                        contenedor.style.display = 'block';
                        btnAsignar.disabled = true;
                    }
                })
                .catch(error => {
                    console.error('Error:', error);
                    areasDiv.innerHTML = '<div class="col-12"><p class="text-danger">Error al cargar las áreas: ' + error.message + '</p></div>';
                    contenedor.style.display = 'block';
                    btnAsignar.disabled = true;
                });
        }

        function verificarSeleccion() {
            const checkboxes = document.querySelectorAll('input[name="areas[]"]:checked');
            const btnAsignar = document.getElementById('btnAsignar');
            btnAsignar.disabled = checkboxes.length === 0;
        }
    </script>
</body>
</html>
