<?php
require_once '../config/db.php';
require_once '../config/functions.php';

header('Content-Type: application/json');

// Obtener acción priorizando GET sobre POST
$accion = '';
if (isset($_GET['accion'])) {
    $accion = $_GET['accion'];
} elseif (isset($_POST['accion'])) {
    $accion = $_POST['accion'];
}

try {
    switch ($accion) {
        // Obtener secciones disponibles por nivel y año lectivo
        case 'get_secciones':
            $id_nivel = $_POST['id_nivel'] ?? 0;
            $id_anio_lectivo = $_POST['id_anio_lectivo'] ?? 0;

            if (!$id_nivel || !$id_anio_lectivo) {
                throw new Exception('Faltan parámetros');
            }

            // Obtener secciones que NO tienen programación activa
            $stmt = $pdo->prepare("
                SELECT s.id, s.grado, s.seccion, s.id_turno, s.capacidad,
                       t.nombre as turno, t.abreviatura as turno_abrev,
                       t.hora_ingreso, t.hora_salida, t.tolerancia_minutos
                FROM secciones s
                INNER JOIN turnos t ON t.id = s.id_turno
                WHERE s.id_nivel = :id_nivel
                AND s.id_anio_lectivo = :id_anio_lectivo
                AND s.id NOT IN (
                    SELECT id_seccion
                    FROM programaciones
                    WHERE id_anio_lectivo = :anio_lectivo2
                    AND estado != 'inactivo'
                )
                ORDER BY s.grado, s.seccion
            ");
            $stmt->execute([
                'id_nivel' => $id_nivel,
                'id_anio_lectivo' => $id_anio_lectivo,
                'anio_lectivo2' => $id_anio_lectivo
            ]);

            echo json_encode([
                'success' => true,
                'secciones' => $stmt->fetchAll()
            ]);
            break;

        // Obtener docentes disponibles (para tutor o áreas)
        case 'get_docentes':
            $stmt = $pdo->query("
                SELECT d.id, d.dni,
                       CONCAT(d.apellido_paterno, ' ', d.apellido_materno, ', ', d.nombres) AS nombre
                FROM docentes d
                WHERE d.estado = 'activo'
                ORDER BY d.apellido_paterno, d.apellido_materno
            ");

            echo json_encode([
                'success' => true,
                'docentes' => $stmt->fetchAll()
            ]);
            break;

        // Obtener áreas por nivel y año lectivo
        case 'get_areas_by_nivel':
            $id_nivel = $_POST['id_nivel'] ?? 0;
            $id_anio_lectivo = getAnioLectivoSeleccionado();

            if (!$id_nivel) {
                throw new Exception('Falta especificar nivel');
            }

            $stmt = $pdo->prepare("
                SELECT aa.id, aa.area
                FROM areas_academicas aa
                WHERE aa.id_nivel = :id_nivel
                AND aa.id_anio_lectivo = :id_anio_lectivo
                AND aa.estado = 'activo'
                ORDER BY aa.area
            ");
            $stmt->execute([
                'id_nivel' => $id_nivel,
                'id_anio_lectivo' => $id_anio_lectivo
            ]);

            echo json_encode([
                'success' => true,
                'areas' => $stmt->fetchAll()
            ]);
            break;

        // Obtener auxiliares disponibles
        case 'get_auxiliares':
            $stmt = $pdo->query("
                SELECT a.id, a.dni,
                       CONCAT(a.apellido_paterno, ' ', a.apellido_materno, ', ', a.nombres) AS nombre
                FROM auxiliares a
                WHERE a.estado = 'activo'
                ORDER BY a.apellido_paterno, a.apellido_materno
            ");

            echo json_encode([
                'success' => true,
                'auxiliares' => $stmt->fetchAll()
            ]);
            break;

        // Obtener estudiantes sin programación en el año lectivo actual
        case 'get_estudiantes_disponibles':
            $id_anio_lectivo = getAnioLectivoSeleccionado();
            $busqueda = $_POST['busqueda'] ?? '';

            $sql = "
                SELECT e.id, e.dni,
                       CONCAT(e.apellido_paterno, ' ', IFNULL(e.apellido_materno, ''), ', ', e.nombres) AS nombre,
                       e.genero
                FROM estudiantes e
                WHERE e.estado = 'activo'
                AND e.id NOT IN (
                    SELECT pe.id_estudiante
                    FROM programaciones_estudiantes pe
                    INNER JOIN programaciones p ON p.id = pe.id_programacion
                    WHERE p.id_anio_lectivo = :id_anio_lectivo
                    AND pe.estado = 'activo'
                )
            ";

            $params = ['id_anio_lectivo' => $id_anio_lectivo];

            if (!empty($busqueda)) {
                $sql .= " AND (
                    e.dni LIKE :busqueda1
                    OR e.apellido_paterno LIKE :busqueda2
                    OR e.apellido_materno LIKE :busqueda3
                    OR e.nombres LIKE :busqueda4
                )";
                $busqueda_param = "%$busqueda%";
                $params['busqueda1'] = $busqueda_param;
                $params['busqueda2'] = $busqueda_param;
                $params['busqueda3'] = $busqueda_param;
                $params['busqueda4'] = $busqueda_param;
            }

            $sql .= " ORDER BY e.apellido_paterno, e.apellido_materno LIMIT 50";

            $stmt = $pdo->prepare($sql);
            $stmt->execute($params);

            echo json_encode([
                'success' => true,
                'estudiantes' => $stmt->fetchAll()
            ]);
            break;

        // Asignar estudiantes a programación
        case 'asignar_estudiantes':
            $id_programacion = $_POST['id_programacion'] ?? 0;
            $estudiantes = $_POST['estudiantes'] ?? [];

            if (!$id_programacion || empty($estudiantes)) {
                throw new Exception('Faltan parámetros');
            }

            $pdo->beginTransaction();

            $stmt = $pdo->prepare("
                INSERT INTO programaciones_estudiantes
                (id_programacion, id_estudiante, fecha_asignacion, estado)
                VALUES (:id_programacion, :id_estudiante, CURDATE(), 'activo')
            ");

            foreach ($estudiantes as $id_estudiante) {
                try {
                    $stmt->execute([
                        'id_programacion' => $id_programacion,
                        'id_estudiante' => $id_estudiante
                    ]);
                } catch (PDOException $e) {
                    // Si ya existe, continuamos
                    if ($e->getCode() != 23000) {
                        throw $e;
                    }
                }
            }

            $pdo->commit();

            echo json_encode([
                'success' => true,
                'mensaje' => count($estudiantes) . ' estudiantes asignados correctamente'
            ]);
            break;

        // Desasignar estudiante de programación
        case 'desasignar_estudiante':
            $id = $_POST['id'] ?? 0;

            if (!$id) {
                throw new Exception('Falta ID');
            }

            $stmt = $pdo->prepare("
                UPDATE programaciones_estudiantes
                SET estado = 'retirado', updated_at = NOW()
                WHERE id = :id
            ");
            $stmt->execute(['id' => $id]);

            echo json_encode([
                'success' => true,
                'mensaje' => 'Estudiante desasignado correctamente'
            ]);
            break;

        // Asignar tutor a programación
        case 'asignar_tutor':
            $id_programacion = $_POST['id_programacion'] ?? 0;
            $id_docente = $_POST['id_docente'] ?? 0;

            if (!$id_programacion || !$id_docente) {
                throw new Exception('Faltan parámetros');
            }

            $stmt = $pdo->prepare("
                UPDATE programaciones
                SET id_tutor = :id_docente, updated_at = NOW()
                WHERE id = :id_programacion
            ");
            $stmt->execute([
                'id_docente' => $id_docente,
                'id_programacion' => $id_programacion
            ]);

            echo json_encode([
                'success' => true,
                'mensaje' => 'Tutor asignado correctamente'
            ]);
            break;

        // Asignar área + docente a programación
        case 'asignar_area':
            $id_programacion = $_POST['id_programacion'] ?? 0;
            $id_docente = $_POST['id_docente'] ?? 0;
            $id_area = $_POST['id_area'] ?? 0;

            if (!$id_programacion || !$id_docente || !$id_area) {
                throw new Exception('Faltan parámetros');
            }

            $stmt = $pdo->prepare("
                INSERT INTO programaciones_docentes_areas
                (id_programacion, id_docente, id_area_academica, fecha_asignacion, estado)
                VALUES (:id_programacion, :id_docente, :id_area, CURDATE(), 'activo')
                ON DUPLICATE KEY UPDATE
                estado = 'activo',
                observaciones = VALUES(observaciones)
            ");
            $stmt->execute([
                'id_programacion' => $id_programacion,
                'id_docente' => $id_docente,
                'id_area' => $id_area
            ]);

            echo json_encode([
                'success' => true,
                'mensaje' => 'Área asignada correctamente'
            ]);
            break;

        // Desasignar área de programación
        case 'desasignar_area':
            $id = $_POST['id'] ?? 0;

            if (!$id) {
                throw new Exception('Falta ID');
            }

            $stmt = $pdo->prepare("
                UPDATE programaciones_docentes_areas
                SET estado = 'inactivo'
                WHERE id = :id
            ");
            $stmt->execute(['id' => $id]);

            echo json_encode([
                'success' => true,
                'mensaje' => 'Área desasignada correctamente'
            ]);
            break;

        // Asignar auxiliar a programación
        case 'asignar_auxiliar':
            $id_programacion = $_POST['id_programacion'] ?? 0;
            $id_auxiliar = $_POST['id_auxiliar'] ?? 0;

            if (!$id_programacion || !$id_auxiliar) {
                throw new Exception('Faltan parámetros');
            }

            $stmt = $pdo->prepare("
                INSERT INTO programaciones_auxiliares
                (id_programacion, id_auxiliar, fecha_asignacion, estado)
                VALUES (:id_programacion, :id_auxiliar, CURDATE(), 'activo')
                ON DUPLICATE KEY UPDATE
                estado = 'activo',
                observaciones = VALUES(observaciones)
            ");
            $stmt->execute([
                'id_programacion' => $id_programacion,
                'id_auxiliar' => $id_auxiliar
            ]);

            echo json_encode([
                'success' => true,
                'mensaje' => 'Auxiliar asignado correctamente'
            ]);
            break;

        // Desasignar auxiliar de programación
        case 'desasignar_auxiliar':
            $id = $_POST['id'] ?? 0;

            if (!$id) {
                throw new Exception('Falta ID');
            }

            $stmt = $pdo->prepare("
                UPDATE programaciones_auxiliares
                SET estado = 'inactivo'
                WHERE id = :id
            ");
            $stmt->execute(['id' => $id]);

            echo json_encode([
                'success' => true,
                'mensaje' => 'Auxiliar desasignado correctamente'
            ]);
            break;

        // Obtener áreas por nivel de la programación
        case 'get_areas_docente':
            $id_anio_lectivo = $_REQUEST['id_anio_lectivo'] ?? 0;
            $id_programacion = $_REQUEST['id_programacion'] ?? 0;

            if (!$id_anio_lectivo || !$id_programacion) {
                throw new Exception('Faltan parámetros');
            }

            // Obtener el nivel de la programación para filtrar áreas
            $stmt = $pdo->prepare("
                SELECT s.id_nivel
                FROM programaciones p
                INNER JOIN secciones s ON s.id = p.id_seccion
                WHERE p.id = ?
            ");
            $stmt->execute([$id_programacion]);
            $prog = $stmt->fetch();

            if (!$prog) {
                throw new Exception('Programación no encontrada');
            }

            $id_nivel = $prog['id_nivel'];

            // Buscar áreas académicas por nivel y año lectivo
            $sql = "
                SELECT aa.id, aa.area, aa.id_nivel, n.nombre as nivel_nombre
                FROM areas_academicas aa
                INNER JOIN niveles n ON n.id = aa.id_nivel
                WHERE aa.id_nivel = :id_nivel
                AND aa.id_anio_lectivo = :id_anio_lectivo
                AND aa.estado = 'activo'
            ";

            // Excluir áreas ya asignadas a esta programación
            $sql .= " AND aa.id NOT IN (
                SELECT id_area_academica
                FROM programaciones_docentes_areas
                WHERE id_programacion = :id_programacion
                AND estado = 'activo'
            )";

            $sql .= " ORDER BY aa.area";

            $stmt = $pdo->prepare($sql);
            $stmt->execute([
                'id_nivel' => $id_nivel,
                'id_anio_lectivo' => $id_anio_lectivo,
                'id_programacion' => $id_programacion
            ]);

            echo json_encode([
                'success' => true,
                'areas' => $stmt->fetchAll()
            ]);
            break;

        // Obtener estadísticas de programación
        case 'get_estadisticas':
            $id_programacion = $_POST['id_programacion'] ?? 0;

            if (!$id_programacion) {
                throw new Exception('Falta ID de programación');
            }

            // Estudiantes
            $stmt = $pdo->prepare("
                SELECT COUNT(*) AS total
                FROM programaciones_estudiantes
                WHERE id_programacion = :id_programacion
                AND estado = 'activo'
            ");
            $stmt->execute(['id_programacion' => $id_programacion]);
            $total_estudiantes = $stmt->fetchColumn();

            // Áreas
            $stmt = $pdo->prepare("
                SELECT COUNT(*) AS total
                FROM programaciones_docentes_areas
                WHERE id_programacion = :id_programacion
                AND estado = 'activo'
            ");
            $stmt->execute(['id_programacion' => $id_programacion]);
            $total_areas = $stmt->fetchColumn();

            // Auxiliares
            $stmt = $pdo->prepare("
                SELECT COUNT(*) AS total
                FROM programaciones_auxiliares
                WHERE id_programacion = :id_programacion
                AND estado = 'activo'
            ");
            $stmt->execute(['id_programacion' => $id_programacion]);
            $total_auxiliares = $stmt->fetchColumn();

            echo json_encode([
                'success' => true,
                'estadisticas' => [
                    'estudiantes' => $total_estudiantes,
                    'areas' => $total_areas,
                    'auxiliares' => $total_auxiliares
                ]
            ]);
            break;

        default:
            echo json_encode([
                'success' => false,
                'mensaje' => 'Acción no válida'
            ]);
            break;
    }

} catch (Exception $e) {
    if (isset($pdo) && $pdo->inTransaction()) {
        $pdo->rollBack();
    }

    echo json_encode([
        'success' => false,
        'mensaje' => $e->getMessage()
    ]);
}
