<?php
require_once '../config/db.php';
require_once '../config/functions.php';

requerirRol(['admin', 'docente', 'auxiliar']);

$errores = [];
$exito = false;

// Obtener datos para selects
$anio_activo = query("SELECT id, anio FROM anios_lectivos WHERE estado = 'activo' LIMIT 1")->fetch();
$tipos_incidencia = query("SELECT * FROM tipos_incidencia WHERE estado = 'activo' ORDER BY gravedad, nombre")->fetchAll();

// Obtener estudiantes activos
$estudiantes = query("SELECT e.id, e.dni,
                      CONCAT(e.apellido_paterno, ' ', e.apellido_materno, ', ', e.nombres) AS nombre_completo,
                      s.grado, s.seccion, n.nombre AS nivel
                      FROM estudiantes e
                      LEFT JOIN secciones s ON e.id_seccion = s.id
                      LEFT JOIN niveles n ON s.id_nivel = n.id
                      WHERE e.id_anio_lectivo = ? AND e.estado = 'activo'
                      ORDER BY e.apellido_paterno, e.apellido_materno",
                      [$anio_activo['id']])->fetchAll();

// Obtener docente actual si es rol docente
$docente_actual = null;
if ($_SESSION['usuario_rol'] === 'docente') {
    $docente_actual = query("SELECT id, nombres, apellido_paterno, apellido_materno FROM docentes WHERE id_usuario = ?", [$_SESSION['usuario_id']])->fetch();
}

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $id_estudiante = $_POST['id_estudiante'] ?? '';
    $id_tipo_incidencia = $_POST['id_tipo_incidencia'] ?? '';
    $fecha = $_POST['fecha'] ?? date('Y-m-d');
    $hora = $_POST['hora'] ?? date('H:i');
    $descripcion = sanitizar($_POST['descripcion'] ?? '');
    $accion_tomada = sanitizar($_POST['accion_tomada'] ?? '');

    // Validaciones
    if (empty($id_estudiante)) $errores[] = 'Debe seleccionar un estudiante';
    if (empty($id_tipo_incidencia)) $errores[] = 'Debe seleccionar el tipo de incidencia';
    if (empty($fecha)) $errores[] = 'La fecha es obligatoria';
    if (empty($hora)) $errores[] = 'La hora es obligatoria';
    if (empty($descripcion)) $errores[] = 'La descripción es obligatoria';

    if (empty($errores)) {
        $id_docente = null;
        if ($_SESSION['usuario_rol'] === 'docente' && $docente_actual) {
            $id_docente = $docente_actual['id'];
        }

        $sql = "INSERT INTO incidencias (id_estudiante, id_tipo_incidencia, id_docente, id_anio_lectivo, fecha, hora, descripcion, accion_tomada)
                VALUES (:id_estudiante, :id_tipo_incidencia, :id_docente, :id_anio_lectivo, :fecha, :hora, :descripcion, :accion_tomada)";

        $params = [
            'id_estudiante' => $id_estudiante,
            'id_tipo_incidencia' => $id_tipo_incidencia,
            'id_docente' => $id_docente,
            'id_anio_lectivo' => $anio_activo['id'],
            'fecha' => $fecha,
            'hora' => $hora . ':00',
            'descripcion' => $descripcion,
            'accion_tomada' => $accion_tomada
        ];

        if (query($sql, $params)) {
            $id_incidencia = query("SELECT LAST_INSERT_ID()")->fetchColumn();

            // Obtener datos del estudiante y apoderado para notificación
            $estudiante_data = query("SELECT e.*, a.id AS id_apoderado, a.email AS email_apoderado,
                                     CONCAT(a.apellido_paterno, ' ', a.apellido_materno) AS apoderado_nombre
                                     FROM estudiantes e
                                     LEFT JOIN apoderados a ON e.id_apoderado = a.id
                                     WHERE e.id = ?", [$id_estudiante])->fetch();

            // Registrar notificación de email (pendiente de enviar)
            if ($estudiante_data && $estudiante_data['id_apoderado']) {
                $tipo_incidencia = query("SELECT nombre FROM tipos_incidencia WHERE id = ?", [$id_tipo_incidencia])->fetchColumn();

                $asunto = "NUEVA INCIDENCIA: " . strtoupper($estudiante_data['nombres'] . ' ' . $estudiante_data['apellido_paterno']);
                $mensaje = "Se ha registrado una nueva incidencia del estudiante:\n\n";
                $mensaje .= "Estudiante: " . $estudiante_data['nombres'] . ' ' . $estudiante_data['apellido_paterno'] . ' ' . $estudiante_data['apellido_materno'] . "\n";
                $mensaje .= "Tipo de Incidencia: " . $tipo_incidencia . "\n";
                $mensaje .= "Fecha: " . $fecha . "\n";
                $mensaje .= "Hora: " . $hora . "\n";
                $mensaje .= "Descripción: " . $descripcion . "\n";
                if ($accion_tomada) {
                    $mensaje .= "Acción Tomada: " . $accion_tomada . "\n";
                }

                query("INSERT INTO notificaciones_email (id_estudiante, id_apoderado, tipo_notificacion, id_referencia, asunto, mensaje, email_enviado, estado_envio)
                      VALUES (?, ?, 'incidencia', ?, ?, ?, ?, 'pendiente')",
                      [$id_estudiante, $estudiante_data['id_apoderado'], $id_incidencia, $asunto, $mensaje, $estudiante_data['email_apoderado']]);
            }

            setAlerta('success', 'Incidencia registrada correctamente. Se enviará notificación al apoderado.');
            header('Location: index.php');
            exit;
        } else {
            $errores[] = 'Error al registrar la incidencia';
        }
    }
}
?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0, maximum-scale=1.0, user-scalable=no">
    <title>SICA - Registrar Incidencia</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.1/font/bootstrap-icons.css" rel="stylesheet">
    <style>
        :root {
            --primary-color: #1e40af;
            --sidebar-width: 280px;
        }
        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            background-color: #f3f4f6;
        }
        .sidebar {
            position: fixed;
            top: 0;
            left: 0;
            height: 100vh;
            width: var(--sidebar-width);
            background: linear-gradient(180deg, #1e3a8a 0%, #1e40af 100%);
            color: white;
            z-index: 1000;
            overflow-y: auto;
        }
        .main-content {
            margin-left: var(--sidebar-width);
            padding: 30px;
        }
        .form-card {
            max-width: 800px;
            margin: 0 auto;
            background: white;
            border-radius: 15px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.05);
            padding: 30px;
        }
        .form-control, .form-select {
            border-radius: 10px;
            padding: 12px 15px;
        }
        .btn-submit {
            background: linear-gradient(135deg, var(--primary-color), #3b82f6);
            border: none;
            border-radius: 10px;
            padding: 12px 30px;
            font-weight: 600;
        }
        @media (max-width: 992px) {
            .sidebar { display: none; }
            .main-content { margin-left: 0; }
        }
    </style>
</head>
<body>
    <nav class="sidebar">
        <?php include '../includes/sidebar.php'; ?>
    </nav>

    <div class="main-content">
        <div class="d-flex justify-content-between align-items-center mb-4">
            <div>
                <h3 class="mb-1">Registrar Incidencia</h3>
                <p class="text-muted mb-0">Registro de inconductas o faltas disciplinarias - <?= $anio_activo['anio'] ?></p>
            </div>
            <a href="index.php" class="btn btn-outline-secondary">
                <i class="bi bi-arrow-left me-2"></i>Volver
            </a>
        </div>

        <div class="form-card">
            <?php if (!empty($errores)): ?>
                <div class="alert alert-danger">
                    <i class="bi bi-exclamation-triangle-fill me-2"></i>
                    <strong>Errores:</strong><br>
                    <?php foreach ($errores as $error): ?>
                        <?= $error ?><br>
                    <?php endforeach; ?>
                </div>
            <?php endif; ?>

            <form method="POST">
                <div class="row g-3">
                    <div class="col-md-6">
                        <label class="form-label">Estudiante <span class="text-danger">*</span></label>
                        <select name="id_estudiante" class="form-select" required>
                            <option value="">Seleccione un estudiante</option>
                            <?php foreach ($estudiantes as $est): ?>
                                <option value="<?= $est['id'] ?>" <?= (($_POST['id_estudiante'] ?? '') == $est['id'] ? 'selected' : '' ?>>
                                    <?= htmlspecialchars($est['nombre_completo']) ?> - <?= $est['nivel'] ?> <?= $est['grado'] ?> "<?= $est['seccion'] ?>"
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>

                    <div class="col-md-6">
                        <label class="form-label">Tipo de Incidencia <span class="text-danger">*</span></label>
                        <select name="id_tipo_incidencia" class="form-select" required>
                            <option value="">Seleccione el tipo</option>
                            <?php foreach ($tipos_incidencia as $ti): ?>
                                <?php
                                $badge_class = match($ti['gravedad']) {
                                    'leve' => 'bg-success',
                                    'moderada' => 'bg-warning',
                                    'grave' => 'bg-danger',
                                    'muy_grave' => 'bg-dark',
                                    default => 'bg-secondary'
                                };
                                ?>
                                <option value="<?= $ti['id'] ?>" data-gravedad="<?= $ti['gravedad'] ?>" <?= (($_POST['id_tipo_incidencia'] ?? '') == $ti['id'] ? 'selected' : '' ?>>
                                    <?= htmlspecialchars($ti['nombre']) ?> (<?= $ti['gravedad'] ?>)
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>

                    <div class="col-md-6">
                        <label class="form-label">Fecha <span class="text-danger">*</span></label>
                        <input type="date" name="fecha" class="form-control" required value="<?= $_POST['fecha'] ?? date('Y-m-d') ?>">
                    </div>

                    <div class="col-md-6">
                        <label class="form-label">Hora <span class="text-danger">*</span></label>
                        <input type="time" name="hora" class="form-control" required value="<?= $_POST['hora'] ?? date('H:i') ?>">
                    </div>

                    <div class="col-12">
                        <label class="form-label">Descripción del Incidente <span class="text-danger">*</span></label>
                        <textarea name="descripcion" class="form-control" rows="4" required
                                  placeholder="Describa detalladamente lo ocurrido..."><?= htmlspecialchars($_POST['descripcion'] ?? '') ?></textarea>
                    </div>

                    <div class="col-12">
                        <label class="form-label">Acción Tomada</label>
                        <textarea name="accion_tomada" class="form-control" rows="2"
                                  placeholder="Describa la medida correctiva o sanción aplicada..."><?= htmlspecialchars($_POST['accion_tomada'] ?? '') ?></textarea>
                    </div>
                </div>

                <hr class="my-4">

                <div class="alert alert-info">
                    <i class="bi bi-info-circle-fill me-2"></i>
                    Se enviará automáticamente una notificación por correo electrónico al apoderado del estudiante.
                </div>

                <div class="d-flex gap-2">
                    <button type="submit" class="btn btn-submit text-white">
                        <i class="bi bi-save-fill me-2"></i>Registrar Incidencia
                    </button>
                    <a href="index.php" class="btn btn-outline-secondary">Cancelar</a>
                </div>
            </form>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        document.querySelector('select[name="id_tipo_incidencia"]').addEventListener('change', function() {
            const selectedOption = this.options[this.selectedIndex];
            const gravedad = selectedOption.dataset.gravedad;

            if (gravedad === 'grave' || gravedad === 'muy_grave') {
                alert('ATENCIÓN: Esta incidencia es de gravedad ' + gravedad.toUpperCase() + '. Asegúrese de documentar adecuadamente el incidente.');
            }
        });
    </script>
</body>
</html>
