<?php
require_once '../config/db.php';
require_once '../config/functions.php';

requerirRol(['admin', 'docente', 'auxiliar']);

$id = intval($_GET['id'] ?? 0);

if (!$id) {
    setAlerta('danger', 'ID de estudiante no válido');
    header('Location: index.php');
    exit;
}

// Obtener estudiante (solo datos personales)
$estudiante = query("SELECT e.*, u.usuario
                    FROM estudiantes e
                    LEFT JOIN usuarios u ON e.id_usuario = u.id
                    WHERE e.id = ?", [$id])->fetch();

if (!$estudiante) {
    setAlerta('danger', 'Estudiante no encontrado');
    header('Location: index.php');
    exit;
}

// Obtener estadísticas de asistencia
$stmt = query("SELECT
                COUNT(*) AS total_dias,
                SUM(CASE WHEN ea.conteo_asistencia = '1' THEN 1 ELSE 0 END) AS asistencias,
                SUM(CASE WHEN a.estado_codigo IN ('Tardanza', 'Tardanza Justificada') THEN 1 ELSE 0 END) AS tardanzas,
                SUM(CASE WHEN a.estado_codigo IN ('Falta Justificada', 'Falta Injustificada') THEN 1 ELSE 0 END) AS faltas,
                SUM(a.minutos_tardanza) AS total_minutos_tardanza
              FROM asistencias a
              INNER JOIN estados_asistencia ea ON a.id_estado_asistencia = ea.id
              WHERE a.id_estudiante = ?", [$id]);
$estadisticas_asistencia = $stmt ? $stmt->fetch() : false;

// Obtener últimas asistencias (usando la vista que tiene información de estados)
$stmt = query("SELECT
                  v.fecha,
                  v.hora_registro,
                  v.estado_codigo,
                  v.estado_nombre,
                  v.estado_abreviatura,
                  v.estado_color,
                  v.estado_icono,
                  v.minutos_tardanza,
                  v.metodo_registro,
                  v.observaciones
               FROM v_asistencias_detalle v
               WHERE v.id_estudiante = ?
               ORDER BY v.fecha DESC, v.hora_registro DESC
               LIMIT 5", [$id]);
$ultimas_asistencias = $stmt ? $stmt->fetchAll() : [];

// Obtener incidencias recientes
try {
    $stmt = query("SELECT i.*, ti.nombre AS tipo_incidencia, ti.gravedad
                  FROM incidencias i
                  JOIN tipos_incidencia ti ON i.id_tipo_incidencia = ti.id
                  WHERE i.id_estudiante = ?
                  ORDER BY i.fecha DESC, i.hora DESC
                  LIMIT 5", [$id]);
    $incidencias = $stmt ? $stmt->fetchAll() : [];
} catch (Exception $e) {
    // Si hay error (tabla no existe o estructura diferente), dejar array vacío
    $incidencias = [];
}
?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0, maximum-scale=1.0, user-scalable=no">
    <title>SICA - Ver Estudiante</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.1/font/bootstrap-icons.css" rel="stylesheet">
    <link href="../assets/css/sica.css?v=5.1" rel="stylesheet">
    <style>
        .profile-header {
            background: linear-gradient(135deg, var(--primary-color), #3b82f6);
            color: white;
            padding: 30px;
            border-radius: 15px;
            margin-bottom: 25px;
        }
        .stat-card {
            background: white;
            border-radius: 12px;
            padding: 20px;
            text-align: center;
            height: 100%;
            transition: transform 0.2s;
        }
        .stat-card:hover {
            transform: translateY(-5px);
            box-shadow: 0 5px 20px rgba(0,0,0,0.1);
        }
        .stat-value {
            font-size: 2rem;
            font-weight: bold;
            color: var(--primary-color);
        }
        .stat-label {
            color: #64748b;
            font-size: 0.9rem;
        }
        .qr-preview {
            text-align: center;
            padding: 20px;
            background: #f8fafc;
            border-radius: 12px;
        }
        .qr-preview img {
            max-width: 200px;
            border: 3px solid var(--primary-color);
            border-radius: 10px;
        }
    </style>
</head>
<body>
    <nav class="sidebar">
        <?php include '../includes/sidebar.php'; ?>
    </nav>

    <div class="main-content">
        <div class="d-flex justify-content-between align-items-center mb-4">
            <div>
                <h3 class="mb-1">Perfil de Estudiante</h3>
                <p class="text-muted mb-0">Información personal del estudiante</p>
            </div>
            <div class="btn-group">
                <a href="descargar_qr.php?id=<?= $estudiante['id'] ?>" class="btn btn-dark">
                    <i class="bi bi-qr-code me-2"></i>Descargar QR
                </a>
                <a href="../reportes/reporte_individual.php?id=<?= $estudiante['id'] ?>" class="btn btn-primary">
                    <i class="bi bi-file-earmark-bar-graph me-2"></i>Ver Reporte
                </a>
                <a href="editar.php?id=<?= $estudiante['id'] ?>" class="btn btn-warning">
                    <i class="bi bi-pencil-fill me-2"></i>Editar
                </a>
                <a href="index.php" class="btn btn-outline-secondary">
                    <i class="bi bi-arrow-left me-2"></i>Volver
                </a>
            </div>
        </div>

        <!-- Header con información básica -->
        <div class="profile-header">
            <div class="row align-items-center">
                <div class="col-md-8">
                    <h2 class="mb-1"><?= htmlspecialchars($estudiante['apellido_paterno'] . ' ' . $estudiante['apellido_materno'] . ', ' . $estudiante['nombres']) ?></h2>
                    <p class="mb-2"><i class="bi bi-credit-card me-2"></i>DNI: <?= htmlspecialchars($estudiante['dni']) ?></p>
                </div>
                <div class="col-md-4 text-end">
                    <?php if ($estudiante['estado'] === 'activo'): ?>
                        <span class="badge bg-success fs-6">Activo</span>
                    <?php else: ?>
                        <span class="badge bg-secondary fs-6"><?= ucfirst($estudiante['estado']) ?></span>
                    <?php endif; ?>
                </div>
            </div>
        </div>

        <!-- Información Personal y QR -->
        <div class="row">
            <div class="col-md-8">
                <div class="content-card">
                    <h5 class="mb-4 pb-2 border-bottom">
                        <i class="bi bi-person-fill me-2"></i>Información Personal
                    </h5>
                    <div class="row g-3">
                        <div class="col-md-6">
                            <label class="text-muted small">DNI</label>
                            <p class="mb-0 fw-bold"><?= htmlspecialchars($estudiante['dni']) ?></p>
                        </div>
                        <div class="col-md-6">
                            <label class="text-muted small">Usuario</label>
                            <p class="mb-0 fw-bold"><?= htmlspecialchars($estudiante['usuario'] ?? 'No asignado') ?></p>
                        </div>
                        <div class="col-md-6">
                            <label class="text-muted small">Fecha de Nacimiento</label>
                            <p class="mb-0 fw-bold"><?= formatearFecha($estudiante['fecha_nacimiento']) ?></p>
                        </div>
                        <div class="col-md-6">
                            <label class="text-muted small">Género</label>
                            <p class="mb-0 fw-bold"><?= $estudiante['genero'] === 'M' ? 'Masculino' : 'Femenino' ?></p>
                        </div>
                        <div class="col-md-6">
                            <label class="text-muted small">Teléfono</label>
                            <p class="mb-0 fw-bold"><?= htmlspecialchars($estudiante['telefono'] ?? 'No registrado') ?></p>
                        </div>
                        <div class="col-md-6">
                            <label class="text-muted small">Fecha de Registro</label>
                            <p class="mb-0 fw-bold"><?= formatearDateTime($estudiante['fecha_registro']) ?></p>
                        </div>
                        <div class="col-12">
                            <label class="text-muted small">Dirección</label>
                            <p class="mb-0 fw-bold"><?= htmlspecialchars($estudiante['direccion'] ?? 'No registrada') ?></p>
                        </div>
                    </div>
                </div>
            </div>

            <div class="col-md-4">
                <div class="content-card">
                    <h5 class="mb-3 text-center">
                        <i class="bi bi-qr-code me-2"></i>Código QR
                    </h5>
                    <div class="qr-preview">
                        <img src="generar_qr.php?dni=<?= urlencode($estudiante['dni']) ?>&formato=png&tamanio=200"
                             alt="Código QR del estudiante">
                        <p class="mt-2 mb-0 small text-muted">
                            <i class="bi bi-info-circle me-1"></i>
                            Escanear para registrar asistencia
                        </p>
                    </div>
                    <a href="descargar_qr.php?id=<?= $estudiante['id'] ?>" class="btn btn-dark w-100 mt-3">
                        <i class="bi bi-download me-2"></i>Descargar QR
                    </a>
                </div>
            </div>
        </div>

        <!-- Estadísticas de Asistencia -->
        <div class="content-card">
            <h5 class="mb-4 pb-2 border-bottom">
                <i class="bi bi-calendar-check me-2"></i>Estadísticas de Asistencia
            </h5>
            <div class="row g-3">
                <div class="col-md-3">
                    <div class="stat-card">
                        <div class="stat-value"><?= $estadisticas_asistencia['total_dias'] ?? 0 ?></div>
                        <div class="stat-label">Días Registrados</div>
                    </div>
                </div>
                <div class="col-md-3">
                    <div class="stat-card" style="border-left: 4px solid #10b981;">
                        <div class="stat-value text-success"><?= $estadisticas_asistencia['asistencias'] ?? 0 ?></div>
                        <div class="stat-label">Asistencias</div>
                    </div>
                </div>
                <div class="col-md-3">
                    <div class="stat-card" style="border-left: 4px solid #f59e0b;">
                        <div class="stat-value text-warning"><?= $estadisticas_asistencia['tardanzas'] ?? 0 ?></div>
                        <div class="stat-label">Tardanzas</div>
                    </div>
                </div>
                <div class="col-md-3">
                    <div class="stat-card" style="border-left: 4px solid #ef4444;">
                        <div class="stat-value text-danger"><?= $estadisticas_asistencia['faltas'] ?? 0 ?></div>
                        <div class="stat-label">Faltas</div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Últimas Asistencias -->
        <?php if (!empty($ultimas_asistencias)): ?>
        <div class="content-card">
            <h5 class="mb-4 pb-2 border-bottom">
                <i class="bi bi-clock-history me-2"></i>Últimas Asistencias
            </h5>
            <div class="table-responsive">
                <table class="table table-sm">
                    <thead>
                        <tr>
                            <th>Fecha</th>
                            <th>Hora</th>
                            <th>Estado</th>
                            <th>Tardanza</th>
                            <th>Método</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($ultimas_asistencias as $asistencia): ?>
                        <tr>
                            <td><?= formatearFecha($asistencia['fecha']) ?></td>
                            <td><?= date('H:i:s', strtotime($asistencia['hora_registro'])) ?></td>
                            <td>
                                <span class="badge" style="background-color: <?= $asistencia['estado_color'] ?>;">
                                    <i class="bi <?= $asistencia['estado_icono'] ?> me-1"></i>
                                    <?= htmlspecialchars($asistencia['estado_abreviatura']) ?>
                                </span>
                            </td>
                            <td><?= $asistencia['minutos_tardanza'] ? $asistencia['minutos_tardanza'] . ' min' : '-' ?></td>
                            <td>
                                <i class="bi bi-<?= $asistencia['metodo_registro'] === 'qr' ? 'qr-code' : 'pencil-fill' ?>"></i>
                                <?= ucfirst($asistencia['metodo_registro']) ?>
                            </td>
                        </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        </div>
        <?php endif; ?>

        <!-- Incidencias Recientes -->
        <?php if (!empty($incidencias)): ?>
        <div class="content-card">
            <h5 class="mb-4 pb-2 border-bottom">
                <i class="bi bi-exclamation-triangle me-2"></i>Incidencias Recientes
            </h5>
            <div class="table-responsive">
                <table class="table table-sm">
                    <thead>
                        <tr>
                            <th>Fecha</th>
                            <th>Tipo</th>
                            <th>Gravedad</th>
                            <th>Descripción</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($incidencias as $inc): ?>
                        <tr>
                            <td><?= formatearFecha($inc['fecha']) ?></td>
                            <td><?= htmlspecialchars($inc['tipo_incidencia']) ?></td>
                            <td>
                                <?php
                                $badge_gravedad = match($inc['gravedad']) {
                                    'leve' => 'success',
                                    'moderada' => 'warning',
                                    'grave' => 'danger',
                                    'muy_grave' => 'dark',
                                    default => 'secondary'
                                };
                                ?>
                                <span class="badge bg-<?= $badge_gravedad ?>">
                                    <?= ucfirst($inc['gravedad']) ?>
                                </span>
                            </td>
                            <td><?= htmlspecialchars(substr($inc['descripcion'], 0, 50)) ?>...</td>
                        </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        </div>
        <?php endif; ?>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>
