<?php
require_once '../config/db.php';
require_once '../config/functions.php';

requerirRol(['admin']);

header('Content-Type: application/json');

$respuesta = [
    'success' => false,
    'mensaje' => '',
    'creados' => 0,
    'errores' => []
];

try {
    if (!isset($_FILES['archivo']) || $_FILES['archivo']['error'] !== UPLOAD_ERR_OK) {
        throw new Exception('No se recibió ningún archivo válido');
    }

    $archivo = $_FILES['archivo'];
    $nombre_archivo = $archivo['name'];
    $tmp_archivo = $archivo['tmp_name'];

    // Verificar extensión
    $extension = strtolower(pathinfo($nombre_archivo, PATHINFO_EXTENSION));
    if ($extension !== 'xlsx') {
        throw new Exception('El archivo debe ser Excel (.xlsx)');
    }

    // Cargar SimpleXLSX
    require_once '../vendor/shuchkin/simplexlsx/src/SimpleXLSX.php';

    if (!class_exists('Shuchkin\SimpleXLSX')) {
        throw new Exception('No se pudo cargar la librería SimpleXLSX.');
    }

    // Parsear archivo Excel
    $xlsx = \Shuchkin\SimpleXLSX::parse($tmp_archivo);

    if (!$xlsx) {
        throw new Exception('Error al leer el archivo Excel: ' . \Shuchkin\SimpleXLSX::parseError());
    }

    // Obtener filas
    $rows = $xlsx->rows();

    if (count($rows) < 2) {
        throw new Exception('El archivo debe tener encabezados y al menos una fila de datos');
    }

    // Mapear encabezados a índices de columna
    $encabezados = $rows[0];
    $columnas = array_flip($encabezados);

    // Validar encabezados requeridos
    $requeridos = ['DNI *', 'Apellido Paterno *', 'Apellido Materno *', 'Nombres *', 'Fecha Nacimiento * (dd/mm/aaaa)', 'Genero * (M/F)'];
    foreach ($requeridos as $req) {
        if (!isset($columnas[$req])) {
            throw new Exception("Falta la columna requerida: $req");
        }
    }

    // Procesar estudiantes (saltar fila de encabezados)
    $estudiantes = [];
    for ($i = 1; $i < count($rows); $i++) {
        $fila = $rows[$i];

        $dni = trim($fila[$columnas['DNI *']] ?? '');
        if (empty($dni)) continue; // Saltar filas vacías

        $estudiantes[] = [
            'dni' => $dni,
            'apellido_paterno' => trim($fila[$columnas['Apellido Paterno *']] ?? ''),
            'apellido_materno' => trim($fila[$columnas['Apellido Materno *']] ?? ''),
            'nombres' => trim($fila[$columnas['Nombres *']] ?? ''),
            'fecha_nacimiento' => trim($fila[$columnas['Fecha Nacimiento * (dd/mm/aaaa)']] ?? ''),
            'genero' => strtoupper(trim($fila[$columnas['Genero * (M/F)']] ?? '')),
            'telefono' => trim($fila[$columnas['Telefono'] ?? ''] ?? ''),
            'direccion' => trim($fila[$columnas['Direccion'] ?? ''] ?? ''),
        ];
    }

    if (empty($estudiantes)) {
        throw new Exception('No se encontraron estudiantes válidos en el archivo');
    }

    // Procesar estudiantes
    $creados = 0;
    $pdo->beginTransaction();

    foreach ($estudiantes as $idx => $est) {
        $linea = $idx + 2; // +2 por encabezados y 0-index

        // Validaciones
        if (empty($est['dni']) || !preg_match('/^\d{8}$/', $est['dni'])) {
            $respuesta['errores'][] = "Línea $linea: DNI inválido ({$est['dni']})";
            continue;
        }

        if (empty($est['apellido_paterno']) || empty($est['apellido_materno']) || empty($est['nombres'])) {
            $respuesta['errores'][] = "Línea $linea: Faltan nombres del estudiante";
            continue;
        }

        if (empty($est['fecha_nacimiento'])) {
            $respuesta['errores'][] = "Línea $linea: Fecha de nacimiento requerida";
            continue;
        }

        if (empty($est['genero']) || !in_array($est['genero'], ['M', 'F'])) {
            $respuesta['errores'][] = "Línea $linea: Género inválido (debe ser M o F)";
            continue;
        }

        // Verificar duplicados
        $existe = query("SELECT id FROM estudiantes WHERE dni = ?", [$est['dni']])->fetch();
        if ($existe) {
            $respuesta['errores'][] = "Línea $linea: El DNI {$est['dni']} ya está registrado";
            continue;
        }

        $existe_usuario = query("SELECT id FROM usuarios WHERE usuario = ?", [$est['dni']])->fetch();
        if ($existe_usuario) {
            $respuesta['errores'][] = "Línea $linea: El DNI {$est['dni']} ya existe como usuario";
            continue;
        }

        // Formato de fecha (asumimos YYYY-MM-DD o DD/MM/YYYY)
        $fecha_nacimiento = $est['fecha_nacimiento'];
        if (strpos($fecha_nacimiento, '/') !== false) {
            $partes = explode('/', $fecha_nacimiento);
            if (count($partes) === 3) {
                $fecha_nacimiento = "$partes[2]-$partes[1]-$partes[0]";
            }
        }

        // Crear usuario
        $password_hash = password_hash($est['dni'], PASSWORD_DEFAULT);
        $stmt = $pdo->prepare("INSERT INTO usuarios (usuario, password, rol, estado) VALUES (?, ?, ?, ?)");
        $stmt->execute([$est['dni'], $password_hash, 'estudiante', 'activo']);
        $id_usuario = $pdo->lastInsertId();

        // Crear estudiante (solo datos personales)
        $stmt = $pdo->prepare("INSERT INTO estudiantes (dni, nombres, apellido_paterno, apellido_materno, fecha_nacimiento, genero, direccion, telefono, id_usuario)
                VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)");
        $stmt->execute([
            $est['dni'],
            $est['nombres'],
            $est['apellido_paterno'],
            $est['apellido_materno'],
            $fecha_nacimiento,
            $est['genero'],
            $est['direccion'],
            $est['telefono'],
            $id_usuario
        ]);

        $creados++;
    }

    $pdo->commit();

    $respuesta['success'] = true;
    $respuesta['mensaje'] = 'Importación completada';
    $respuesta['creados'] = $creados;

} catch (Exception $e) {
    if (isset($pdo) && $pdo->inTransaction()) {
        $pdo->rollBack();
    }
    $respuesta['mensaje'] = $e->getMessage();
}

echo json_encode($respuesta);
