-- ============================================
-- SICA - Migración: Normalizar tabla de Turnos
-- ============================================
-- Cambio: Crear tabla turnos independiente con horarios

USE sica2025;

-- PASO 1: Crear tabla turnos
CREATE TABLE IF NOT EXISTS turnos (
    id INT AUTO_INCREMENT PRIMARY KEY,
    nombre VARCHAR(50) NOT NULL UNIQUE COMMENT 'Nombre del turno: mañana, tarde, noche',
    abreviatura VARCHAR(20) NOT NULL UNIQUE COMMENT 'Abreviatura: MAÑ, TAR, NOC',
    hora_ingreso TIME NOT NULL,
    hora_salida TIME NOT NULL,
    tolerancia_minutos INT DEFAULT 15 COMMENT 'Tolerancia en minutos para tardanzas',
    estado ENUM('activo', 'inactivo') DEFAULT 'activo',
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    updated_at DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    INDEX idx_estado (estado),
    INDEX idx_abreviatura (abreviatura)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci COMMENT='Turnos con horarios';

-- PASO 2: Insertar turnos iniciales
INSERT INTO turnos (nombre, abreviatura, hora_ingreso, hora_salida, tolerancia_minutos) VALUES
('Mañana', 'MAÑ', '08:00:00', '12:00:00', 15),
('Tarde', 'TAR', '13:00:00', '18:00:00', 15)
ON DUPLICATE KEY UPDATE updated_at = CURRENT_TIMESTAMP;

-- PASO 3: Verificar turnos creados
SELECT * FROM turnos;

-- PASO 4: Agregar campo id_turno en secciones (sin eliminar el campo viejo aún)
ALTER TABLE secciones ADD COLUMN id_turno INT NULL AFTER turno;
ALTER TABLE secciones ADD INDEX idx_id_turno (id_turno);
ALTER TABLE secciones ADD CONSTRAINT fk_secciones_turno FOREIGN KEY (id_turno) REFERENCES turnos(id) ON DELETE SET NULL ON UPDATE CASCADE;

-- PASO 5: Agregar campo id_turno en programaciones
ALTER TABLE programaciones ADD COLUMN id_turno INT NULL AFTER turno;
ALTER TABLE programaciones ADD INDEX idx_id_turno (id_turno);
ALTER TABLE programaciones ADD CONSTRAINT fk_programaciones_turno FOREIGN KEY (id_turno) REFERENCES turnos(id) ON DELETE SET NULL ON UPDATE CASCADE;

-- PASO 6: Migrar datos existentes a la nueva estructura
-- Para secciones: mapear 'mañana' -> 1, 'tarde' -> 2
UPDATE secciones SET id_turno = CASE turno
    WHEN 'mañana' THEN 1
    WHEN 'tarde' THEN 2
    ELSE NULL
END;

-- Para programaciones: mapear 'mañana' -> 1, 'tarde' -> 2
UPDATE programaciones SET id_turno = CASE turno
    WHEN 'mañana' THEN 1
    WHEN 'tarde' THEN 2
    ELSE NULL
END;

-- PASO 7: Verificar migración
SELECT
    s.id,
    s.grado,
    s.seccion,
    s.turno as turno_viejo,
    s.id_turno,
    t.nombre as turno_nuevo,
    t.hora_ingreso,
    t.hora_salida
FROM secciones s
LEFT JOIN turnos t ON t.id = s.id_turno
LIMIT 5;

SELECT
    p.id,
    s.grado,
    s.seccion,
    p.turno as turno_viejo,
    p.id_turno,
    t.nombre as turno_nuevo,
    t.hora_ingreso,
    t.hora_salida
FROM programaciones p
INNER JOIN secciones s ON s.id = p.id_seccion
LEFT JOIN turnos t ON t.id = p.id_turno
LIMIT 5;

-- NOTA: Los campos 'turno' viejos se pueden eliminar después
-- de verificar que todo funciona correctamente
