-- ============================================
-- MIGRACIÓN: Tabla calendario
-- Fecha: 2025-12-28
-- Descripción: Registro de feriados, vacaciones y días no laborables
-- ============================================

-- Crear tabla calendario
CREATE TABLE IF NOT EXISTS calendario (
    id INT AUTO_INCREMENT PRIMARY KEY,
    fecha DATE NOT NULL,
    tipo ENUM('feriado', 'vacacion', 'dia_no_laboral', 'fin_semana_largo') NOT NULL,
    nombre VARCHAR(100) NOT NULL,
    descripcion TEXT,
    aniolectivo_id INT NULL,
    afecta_todos_niveles BOOLEAN DEFAULT TRUE,
    niveles_afectados VARCHAR(255) NULL,  -- JSON: [1,2,3] para Inicial, Primaria, Secundaria
    observaciones TEXT,
    estado ENUM('activo', 'inactivo') DEFAULT 'activo',
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    updated_at DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    UNIQUE KEY unique_fecha (fecha),
    INDEX idx_tipo (tipo),
    INDEX idx_fecha (fecha),
    INDEX idx_aniolectivo (aniolectivo_id),
    INDEX idx_estado (estado)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================
-- DATOS INICIALES - Feriados Nacionales GLOBALES
-- ============================================
-- NOTA IMPORTANTE: Estos feriados tienen aniolectivo_id = NULL
-- Esto significa que aplican a TODOS los años lectivos (2025, 2026, 2027, etc.)
-- Los feriados nacionales se repiten cada año en la misma fecha

-- Feriados nacionales fijos de Perú (GLOBALES - aplican a todos los años)
INSERT INTO calendario (fecha, tipo, nombre, descripcion, aniolectivo_id, afecta_todos_niveles) VALUES
('2025-01-01', 'feriado', 'Año Nuevo', 'Primer día del año', NULL, TRUE),
('2025-03-29', 'dia_no_laboral', 'Viernes Santo', 'Viernes Santo (Semana Santa)', NULL, TRUE),
('2025-04-10', 'feriado', 'Jueves Santo', 'Jueves Santo (Semana Santa)', NULL, TRUE),
('2025-05-01', 'feriado', 'Día del Trabajo', 'Feriado laboral nacional', NULL, TRUE),
('2025-06-29', 'feriado', 'San Pedro y San Pablo', 'Feriado religioso', NULL, TRUE),
('2025-07-28', 'feriado', 'Fiestas Patrias - Día 1', 'Independencia del Perú', NULL, TRUE),
('2025-07-29', 'feriado', 'Fiestas Patrias - Día 2', 'Gran Parada Militar', NULL, TRUE),
('2025-08-30', 'feriado', 'Santa Rosa de Lima', 'Feriado religioso', NULL, TRUE),
('2025-10-08', 'feriado', 'Combate de Angamos', 'Feriado nacional', NULL, TRUE),
('2025-11-01', 'feriado', 'Todos los Santos', 'Feriado religioso', NULL, TRUE),
('2025-12-08', 'feriado', 'Inmaculada Concepción', 'Feriado religioso', NULL, TRUE),
('2025-12-25', 'feriado', 'Navidad', 'Natividad del Señor', NULL, TRUE);

-- ============================================
-- VISTA: v_calendario_detalle
-- ============================================

CREATE OR REPLACE VIEW v_calendario_detalle AS
SELECT
    c.id,
    c.fecha,
    DAYNAME(c.fecha) AS dia_semana,
    DAY(c.fecha) AS dia,
    MONTH(c.fecha) AS mes,
    YEAR(c.fecha) AS anio,
    c.tipo,
    c.nombre,
    c.descripcion,
    c.aniolectivo_id,
    an.anio AS anio_lectivo,
    c.afecta_todos_niveles,
    c.niveles_afectados,
    c.observaciones,
    c.estado,
    CASE DAYNAME(c.fecha)
        WHEN 'Saturday' THEN 1
        WHEN 'Sunday' THEN 1
        ELSE 0
    END AS es_fin_de_semana
FROM calendario c
LEFT JOIN anios_lectivos an ON c.aniolectivo_id = an.id;

-- ============================================
-- ACTUALIZACIÓN: Feriados Globales para Años Existentes
-- ============================================
-- Si ya ejecutó esta migración anteriormente, ejecute este UPDATE
-- para convertir los feriados existentes en globales:

-- UPDATE calendario SET aniolectivo_id = NULL
-- WHERE tipo IN ('feriado', 'dia_no_laboral') AND afecta_todos_niveles = TRUE;

-- ============================================
-- NOTAS DE MIGRACIÓN
-- ============================================
--
-- Esta tabla permitirá:
-- 1. Registrar feriados nacionales y regionales
-- 2. Configurar periodos de vacaciones por año lectivo
-- 3. Marcar días no laborables por nivel
-- 4. El sistema NO registrará asistencia en estos días
--
-- IMPORTANTÍSIMO - aniolectivo_id:
-- - NULL = Feriado GLOBAL (aplica a TODOS los años lectivos)
-- - ID específico = Vacación/evento específico de ese año lectivo
--
-- Tipos de calendario:
-- - feriado: Días festivos (nacionales, religiosos, cívicos) - Generalmente GLOBALES
-- - vacacion: Periodos de vacaciones escolares - Por AÑO LECTIVO
-- - dia_no_laboral: Días de suspensión de actividades - Generalmente GLOBALES
-- - fin_semana_largo: Fines de semana extendidos (3 días) - Generalmente GLOBALES
--
-- VACACIONES POR AÑO LECTIVO:
-- Las vacaciones escolares se configuran por año lectivo usando las fechas
-- de inicio y fin de cada año lectivo (tabla anios_lectivos).
-- El sistema valida automáticamente si la fecha actual está fuera del rango
-- del año lectivo activo (ej: enero-febrero son vacaciones en Perú).
--
-- Ejemplo de configuración de vacaciones para 2025:
-- INSERT INTO calendario (fecha, tipo, nombre, aniolectivo_id, afecta_todos_niveles)
-- VALUES
-- ('2025-01-15', 'vacacion', 'Inicio de clases 2025', 1, TRUE),
-- ('2025-07-27', 'vacacion', 'Inicio vacaciones de julio', 1, TRUE),
-- ('2025-08-25', 'vacacion', 'Fin de vacaciones de julio', 1, TRUE);
--
-- La lógica de registro de asistencia debe:
-- - Verificar si la fecha actual existe en calendario (GLOBAL o específico del año)
-- - Verificar si la fecha está fuera del rango del año lectivo activo (vacaciones automáticas)
-- - Si es feriado/día no laboral/vacaciones, mostrar advertencia
-- - Permitir registro manual opcional con justificación
--
-- ============================================
