<?php
/**
 * Script para exportar la base de datos SICA2025
 * Corrige problemas con CURDATE(), ENUM y otros para producción
 */

// Función auxiliar para extraer nombre de la vista
function extractViewName($sql) {
    // Buscar patrón simple: `v_nombre`
    if (preg_match('/`(v_\w+)`/i', $sql, $matches)) {
        return "`{$matches[1]}`";
    }
    return '';
}

// Configuración
define('DB_HOST', 'localhost');
define('DB_NAME', 'SICA2025');
define('DB_USER', 'root');
define('DB_PASS', '');
define('DB_CHARSET', 'utf8mb4');

// Archivo de salida
$archivo_salida = __DIR__ . '/SICA2025_PRODUCCION_' . date('Y-m-d_His') . '.sql';

try {
    $pdo = new PDO("mysql:host=" . DB_HOST . ";dbname=" . DB_NAME . ";charset=" . DB_CHARSET, DB_USER, DB_PASS);
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

    // Buffer de salida
    $sql = '';
    $vistas = []; // Para guardar las vistas y crearlas al final

    // Cabecera del archivo
    $sql .= "-- =============================================\n";
    $sql .= "-- SICA - Sistema Integrado de Convivencia y Asistencia\n";
    $sql .= "-- Base de datos: SICA2025\n";
    $sql .= "-- Fecha de exportación: " . date('Y-m-d H:i:s') . "\n";
    $sql .= "-- Compatible con MySQL 5.7+ y MariaDB 10.2+\n";
    $sql .= "-- =============================================\n\n";

    $sql .= "SET SQL_MODE = \"NO_AUTO_VALUE_ON_ZERO\";\n";
    $sql .= "SET time_zone = \"+00:00\";\n";
    $sql .= "SET NAMES utf8mb4;\n\n";

    $sql .= "DROP DATABASE IF EXISTS `SICA2025`;\n";
    $sql .= "CREATE DATABASE `SICA2025` CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;\n";
    $sql .= "USE `SICA2025`;\n\n";

    // Obtener todas las tablas
    $tablas = $pdo->query("SHOW TABLES")->fetchAll(PDO::FETCH_COLUMN);

    echo "Encontradas " . count($tablas) . " tablas\n";
    echo "Exportando...\n";

    foreach ($tablas as $tabla) {
        echo "  - Exportando tabla: $tabla\n";

        // 1. DROP TABLE
        $sql .= "--\n-- Estructura de tabla `$tabla`\n--\n";
        $sql .= "DROP TABLE IF EXISTS `$tabla`;\n";

        // 2. CREATE TABLE o CREATE VIEW
        $create = $pdo->query("SHOW CREATE TABLE `$tabla`")->fetch(PDO::FETCH_ASSOC);

        // Verificar si es una vista o tabla
        if (isset($create['Create View'])) {
            // Es una VISTA - guardar para crearla al final
            $view_sql = $create['Create View'];
            $view_name = extractViewName($view_sql);
            $vistas[] = [
                'name' => $view_name,
                'sql' => $view_sql
            ];
            $sql .= "-- (VISTA - se creará al final del archivo)\n\n";
            continue;
        } elseif (isset($create['Create Table'])) {
            // Es una TABLA
            $create_table = $create['Create Table'];

            // Eliminar restricción de autoincrement para que sea limpia
            $create_table = preg_replace('/AUTO_INCREMENT=\d+/', '', $create_table);

            $sql .= $create_table . ";\n\n";
        } else {
            $sql .= "-- (No se pudo obtener estructura)\n\n";
            continue;
        }

        // 3. INSERT DATA
        $filas = $pdo->query("SELECT COUNT(*) FROM `$tabla`")->fetchColumn();
        if ($filas > 0) {
            $sql .= "--\n-- Volcado de datos para tabla `$tabla`\n--\n";
            $sql .= "LOCK TABLES `$tabla` WRITE;\n";

            // Obtener todas las filas
            $stmt = $pdo->query("SELECT * FROM `$tabla`");
            $describe = $pdo->query("DESCRIBE `$tabla`")->fetchAll(PDO::FETCH_ASSOC);
            $columnas = array_column($describe, 'Field');

            $insert_base = "INSERT INTO `$tabla` (`" . implode('`, `', $columnas) . "`) VALUES\n";

            $values = [];
            $batch_count = 0;
            $batch_size = 100;

            while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
                $valores = [];
                foreach ($row as $valor) {
                    if ($valor === null) {
                        $valores[] = 'NULL';
                    } elseif (is_numeric($valor)) {
                        $valores[] = $valor;
                    } else {
                        // Escapar comillas y caracteres especiales
                        $valor = addslashes($valor);
                        $valores[] = "'$valor'";
                    }
                }
                $values[] = "(" . implode(', ', $valores) . ")";
                $batch_count++;

                if ($batch_count >= $batch_size) {
                    $sql .= $insert_base . implode(",\n", $values) . ";\n";
                    $values = [];
                    $batch_count = 0;
                }
            }

            // Insertar lo que reste
            if (!empty($values)) {
                $sql .= $insert_base . implode(",\n", $values) . ";\n";
            }

            $sql .= "UNLOCK TABLES;\n\n";
        }
    }

    // 4. Crear vistas al final (después de todas las tablas)
    if (!empty($vistas)) {
        $sql .= "-- =============================================\n";
        $sql .= "-- VISTAS (creadas después de las tablas)\n";
        $sql .= "-- =============================================\n\n";

        foreach ($vistas as $vista) {
            $sql .= "DROP VIEW IF EXISTS {$vista['name']};\n";
            $sql .= $vista['sql'] . ";\n\n";
        }
    }

    // Guardar archivo
    file_put_contents($archivo_salida, $sql);

    // Compresión GZIP
    $archivo_gz = $archivo_salida . '.gz';
    $gz = gzopen($archivo_gz, 'w9');
    gzwrite($gz, file_get_contents($archivo_salida));
    gzclose($gz);

    $tamano_original = filesize($archivo_salida);
    $tamano_comprimido = filesize($archivo_gz);

    echo "\n✓ Exportación completada con éxito\n";
    echo "✓ Archivo SQL: $archivo_salida (" . number_format($tamano_original/1024, 2) . " KB)\n";
    echo "✓ Archivo GZ: $archivo_gz (" . number_format($tamano_comprimido/1024, 2) . " KB)\n";
    echo "✓ Compresión: " . number_format((1 - $tamano_comprimido/$tamano_original) * 100, 1) . "%\n";

} catch (PDOException $e) {
    die("Error: " . $e->getMessage());
}
