<?php
require_once '../config/db.php';
require_once '../config/functions.php';
requerirAutenticacion();

// Obtener turnos
$turnos = query("SELECT * FROM turnos ORDER BY FIELD(nombre, 'Mañana', 'Tarde', 'Noche')")->fetchAll();

$errores = [];
$exito = false;

// Procesar formulario
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $nombre = $_POST['nombre'] ?? '';
    $abreviatura = $_POST['abreviatura'] ?? '';
    $hora_ingreso = $_POST['hora_ingreso'] ?? '';
    $hora_salida = $_POST['hora_salida'] ?? '';
    $tolerancia_minutos = $_POST['tolerancia_minutos'] ?? 15;
    $estado = $_POST['estado'] ?? 'activo';

    // Validaciones
    if (!$nombre || !$abreviatura || !$hora_ingreso || !$hora_salida) {
        $errores[] = "Todos los campos son obligatorios";
    }

    if (empty($errores)) {
        try {
            if (isset($_POST['id'])) {
                // Editar
                $id = $_POST['id'];
                $stmt = $pdo->prepare("
                    UPDATE turnos
                    SET nombre = ?, abreviatura = ?, hora_ingreso = ?, hora_salida = ?,
                        tolerancia_minutos = ?, estado = ?, updated_at = NOW()
                    WHERE id = ?
                ");
                $stmt->execute([$nombre, $abreviatura, $hora_ingreso, $hora_salida, $tolerancia_minutos, $estado, $id]);
            } else {
                // Crear
                $stmt = $pdo->prepare("
                    INSERT INTO turnos (nombre, abreviatura, hora_ingreso, hora_salida, tolerancia_minutos, estado)
                    VALUES (?, ?, ?, ?, ?, ?)
                ");
                $stmt->execute([$nombre, $abreviatura, $hora_ingreso, $hora_salida, $tolerancia_minutos, $estado]);
            }

            // Recargar datos de turnos después de guardar/editar
            $turnos = query("SELECT * FROM turnos ORDER BY FIELD(nombre, 'Mañana', 'Tarde', 'Noche')")->fetchAll();

            $exito = true;

        } catch (Exception $e) {
            $errores[] = $e->getMessage();
        }
    }
}
?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0, maximum-scale=1.0, user-scalable=no">
    <title>Gestión de Turnos - SICA</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.0/font/bootstrap-icons.css" rel="stylesheet">
    <link href="../assets/css/sica.css?v=5.1" rel="stylesheet">
</head>
<body>
    <?php include '../includes/layout_elems.php'; ?>

        <div class="d-flex justify-content-between align-items-center mb-4">
            <div class="d-flex align-items-center">
                <button class="btn btn-link d-lg-none me-3" onclick="toggleSidebarMobile()">
                    <i class="bi bi-list fs-4"></i>
                </button>
                <div>
                    <h3 class="mb-0">Gestión de Turnos</h3>
                    <p class="text-muted mb-0">Administración de turnos y horarios</p>
                </div>
            </div>
        </div>

        <?php if ($exito): ?>
            <div class="alert alert-success">
                <i class="bi bi-check-circle me-2"></i>
                Operación realizada correctamente.
            </div>
        <?php endif; ?>

        <?php if (!empty($errores)): ?>
            <div class="alert alert-danger">
                <i class="bi bi-exclamation-triangle me-2"></i>
                <ul class="mb-0 mt-2">
                    <?php foreach ($errores as $error): ?>
                        <li><?= htmlspecialchars($error) ?></li>
                    <?php endforeach; ?>
                </ul>
            </div>
        <?php endif; ?>

        <div class="row">
            <div class="col-lg-6">
                <div class="card border-0 shadow-sm mb-4">
                    <div class="card-header bg-primary text-white">
                        <h6 class="mb-0"><i class="bi bi-plus-circle me-2"></i>Nuevo Turno</h6>
                    </div>
                    <div class="card-body">
                        <form method="POST">
                            <div class="mb-3">
                                <label class="form-label fw-semibold">Nombre del Turno <span class="text-danger">*</span></label>
                                <input type="text" name="nombre" class="form-control"
                                       placeholder="Ej: Mañana, Tarde, Noche" required>
                                <small class="text-muted">Nombre completo del turno</small>
                            </div>

                            <div class="mb-3">
                                <label class="form-label fw-semibold">Abreviatura <span class="text-danger">*</span></label>
                                <input type="text" name="abreviatura" class="form-control"
                                       placeholder="Ej: MAÑ, TAR, NOC" maxlength="5" required>
                                <small class="text-muted">Código corto de máximo 5 caracteres</small>
                            </div>

                            <div class="row">
                                <div class="col-md-6 mb-3">
                                    <label class="form-label fw-semibold">Hora Ingreso <span class="text-danger">*</span></label>
                                    <input type="time" name="hora_ingreso" class="form-control" required>
                                </div>
                                <div class="col-md-6 mb-3">
                                    <label class="form-label fw-semibold">Hora Salida <span class="text-danger">*</span></label>
                                    <input type="time" name="hora_salida" class="form-control" required>
                                </div>
                            </div>

                            <div class="row">
                                <div class="col-md-6 mb-3">
                                    <label class="form-label fw-semibold">Tolerancia (min)</label>
                                    <input type="number" name="tolerancia_minutos" class="form-control"
                                           value="15" min="0" max="60">
                                    <small class="text-muted">Minutos de tolerancia para tardanzas</small>
                                </div>
                                <div class="col-md-6 mb-3">
                                    <label class="form-label fw-semibold">Estado <span class="text-danger">*</span></label>
                                    <select name="estado" class="form-select" required>
                                        <option value="activo">Activo</option>
                                        <option value="inactivo">Inactivo</option>
                                    </select>
                                </div>
                            </div>

                            <button type="submit" class="btn btn-primary">
                                <i class="bi bi-check-circle me-2"></i>Guardar Turno
                            </button>
                        </form>
                    </div>
                </div>
            </div>

            <div class="col-lg-6">
                <div class="card border-0 shadow-sm">
                    <div class="card-header bg-success text-white">
                        <h6 class="mb-0"><i class="bi bi-list-check me-2"></i>Turnos Registrados</h6>
                    </div>
                    <div class="card-body">
                        <?php if (empty($turnos)): ?>
                            <div class="text-center py-4">
                                <i class="bi bi-inbox fs-1 text-muted"></i>
                                <p class="text-muted mt-2">No hay turnos registrados</p>
                            </div>
                        <?php else: ?>
                            <div class="table-responsive">
                                <table class="table table-hover">
                                    <thead class="table-light">
                                        <tr>
                                            <th>Nombre</th>
                                            <th>Abrev.</th>
                                            <th>Horario</th>
                                            <th>Tol.</th>
                                            <th>Estado</th>
                                            <th>Acciones</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php foreach ($turnos as $turno): ?>
                                            <tr>
                                                <td><strong><?= htmlspecialchars($turno['nombre']) ?></strong></td>
                                                <td><code><?= htmlspecialchars($turno['abreviatura']) ?></code></td>
                                                <td>
                                                    <small class="text-muted">
                                                        <?= date('H:i', strtotime($turno['hora_ingreso'])) ?>
                                                        -
                                                        <?= date('H:i', strtotime($turno['hora_salida'])) ?>
                                                    </small>
                                                </td>
                                                <td><?= $turno['tolerancia_minutos'] ?> min</td>
                                                <td>
                                                    <span class="badge bg-<?= $turno['estado'] == 'activo' ? 'success' : 'secondary' ?>">
                                                        <?= ucfirst($turno['estado']) ?>
                                                    </span>
                                                </td>
                                                <td>
                                                    <button class="btn btn-sm btn-outline-primary"
                                                            onclick="editarTurno(<?= htmlspecialchars(json_encode($turno)) ?>)">
                                                        <i class="bi bi-pencil"></i>
                                                    </button>
                                                    <a href="eliminar_turno.php?id=<?= $turno['id'] ?>"
                                                       class="btn btn-sm btn-outline-danger"
                                                       onclick="return confirm('¿Eliminar este turno?')">
                                                        <i class="bi bi-trash"></i>
                                                    </a>
                                                </td>
                                            </tr>
                                        <?php endforeach; ?>
                                    </tbody>
                                </table>
                            </div>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
        </div>

    <?php include '../includes/layout_scripts.php'; ?>

    <script>
    function editarTurno(turno) {
        const form = document.querySelector('form');
        form.nombre.value = turno.nombre;
        form.abreviatura.value = turno.abreviatura;
        form.hora_ingreso.value = turno.hora_ingreso;
        form.hora_salida.value = turno.hora_salida;
        form.tolerancia_minutos.value = turno.tolerancia_minutos;
        form.estado.value = turno.estado;

        // Agregar campo ID
        let idInput = document.querySelector('input[name="id"]');
        if (!idInput) {
            idInput = document.createElement('input');
            idInput.type = 'hidden';
            idInput.name = 'id';
            form.appendChild(idInput);
        }
        idInput.value = turno.id;

        // Cambiar texto del botón
        const submitBtn = form.querySelector('button[type="submit"]');
        submitBtn.innerHTML = '<i class="bi bi-save me-2"></i>Actualizar Turno';

        // Agregar botón cancelar
        let cancelBtn = document.querySelector('#cancelarBtn');
        if (!cancelBtn) {
            cancelBtn = document.createElement('a');
            cancelBtn.id = 'cancelarBtn';
            cancelBtn.href = '#';
            cancelBtn.className = 'btn btn-outline-secondary ms-2';
            cancelBtn.innerHTML = '<i class="bi bi-x-circle me-2"></i>Cancelar';
            cancelBtn.onclick = function(e) {
                e.preventDefault();
                location.reload();
            };
            submitBtn.parentElement.appendChild(cancelBtn);
        }
    }
    </script>
</body>
</html>
