<?php
/**
 * GENERADOR DE FERIADOS GLOBALES
 * Calcula automáticamente los feriados de Perú para múltiples años
 * y los inserta en la base de datos
 *
 * Uso: php generar_calendario_anual.php [anio_inicio] [anio_fin]
 * Ejemplo: php generar_calendario_anual.php 2025 2030
 */

require_once 'db.php';

// Años a generar (por defecto: del año actual al año actual + 5)
$anio_inicio = isset($argv[1]) ? (int)$argv[1] : (int)date('Y');
$anio_fin = isset($argv[2]) ? (int)$argv[2] : $anio_inicio + 5;

echo "=== GENERADOR DE FERIADOS GLOBALES PARA SICA ===\n";
echo "Generando feriados del {$anio_inicio} al {$anio_fin}...\n\n";

// Eliminar feriados globales existentes
echo "1. Eliminando feriados globales existentes...\n";
query("DELETE FROM calendario WHERE aniolectivo_id IS NULL");
echo "   ✓ Feriados antiguos eliminados\n\n";

$feriados_insertados = 0;
$feriados_errores = 0;

// Generar feriados para cada año
for ($anio = $anio_inicio; $anio <= $anio_fin; $anio++) {
    echo "2. Generando feriados del año {$anio}...\n";

    $feriados = obtenerFeriadosPeru($anio);

    foreach ($feriados as $feriado) {
        try {
            $sql = "INSERT INTO calendario (fecha, tipo, nombre, descripcion, aniolectivo_id, afecta_todos_niveles)
                    VALUES (?, ?, ?, ?, NULL, TRUE)";
            query($sql, [
                $feriado['fecha'],
                $feriado['tipo'],
                $feriado['nombre'],
                $feriado['descripcion']
            ]);
            $feriados_insertados++;
            echo "   ✓ {$feriado['fecha']} - {$feriado['nombre']}\n";
        } catch (Exception $e) {
            $feriados_errores++;
            echo "   ✗ Error al insertar {$feriado['nombre']} del {$anio}: {$e->getMessage()}\n";
        }
    }

    echo "\n";
}

echo "\n=== RESUMEN ===\n";
echo "Feriados insertados: {$feriados_insertados}\n";
echo "Errores: {$feriados_errores}\n";
echo "\n✓ Proceso completado.\n";

/**
 * Calcula los feriados de Perú para un año específico
 *
 * @param int $anio Año a calcular
 * @return array Array de feriados con fecha, tipo, nombre, descripcion
 */
function obtenerFeriadosPeru($anio) {
    $feriados = [];

    // Año Nuevo (1 de enero)
    $feriados[] = [
        'fecha' => "{$anio}-01-01",
        'tipo' => 'feriado',
        'nombre' => 'Año Nuevo',
        'descripcion' => 'Primer día del año'
    ];

    // Semana Santa (calcular fechas)
    $semana_santa = calcularSemanaSanta($anio);
    $feriados[] = [
        'fecha' => $semana_santa['jueves_santo'],
        'tipo' => 'feriado',
        'nombre' => 'Jueves Santo',
        'descripcion' => 'Jueves Santo (Semana Santa)'
    ];
    $feriados[] = [
        'fecha' => $semana_santa['viernes_santo'],
        'tipo' => 'dia_no_laboral',
        'nombre' => 'Viernes Santo',
        'descripcion' => 'Viernes Santo (Semana Santa)'
    ];

    // Día del Trabajo (1 de mayo)
    $feriados[] = [
        'fecha' => "{$anio}-05-01",
        'tipo' => 'feriado',
        'nombre' => 'Día del Trabajo',
        'descripcion' => 'Feriado laboral nacional'
    ];

    // San Pedro y San Pablo (29 de junio)
    $feriados[] = [
        'fecha' => "{$anio}-06-29",
        'tipo' => 'feriado',
        'nombre' => 'San Pedro y San Pablo',
        'descripcion' => 'Feriado religioso'
    ];

    // Fiestas Patrias (28 y 29 de julio)
    $feriados[] = [
        'fecha' => "{$anio}-07-28",
        'tipo' => 'feriado',
        'nombre' => 'Fiestas Patrias - Día 1',
        'descripcion' => 'Independencia del Perú'
    ];
    $feriados[] = [
        'fecha' => "{$anio}-07-29",
        'tipo' => 'feriado',
        'nombre' => 'Fiestas Patrias - Día 2',
        'descripcion' => 'Gran Parada Militar'
    ];

    // Santa Rosa de Lima (30 de agosto)
    $feriados[] = [
        'fecha' => "{$anio}-08-30",
        'tipo' => 'feriado',
        'nombre' => 'Santa Rosa de Lima',
        'descripcion' => 'Feriado religioso'
    ];

    // Combate de Angamos (8 de octubre)
    $feriados[] = [
        'fecha' => "{$anio}-10-08",
        'tipo' => 'feriado',
        'nombre' => 'Combate de Angamos',
        'descripcion' => 'Feriado nacional'
    ];

    // Todos los Santos (1 de noviembre)
    $feriados[] = [
        'fecha' => "{$anio}-11-01",
        'tipo' => 'feriado',
        'nombre' => 'Todos los Santos',
        'descripcion' => 'Feriado religioso'
    ];

    // Inmaculada Concepción (8 de diciembre)
    $feriados[] = [
        'fecha' => "{$anio}-12-08",
        'tipo' => 'feriado',
        'nombre' => 'Inmaculada Concepción',
        'descripcion' => 'Feriado religioso'
    ];

    // Navidad (25 de diciembre)
    $feriados[] = [
        'fecha' => "{$anio}-12-25",
        'tipo' => 'feriado',
        'nombre' => 'Navidad',
        'descripcion' => 'Natividad del Señor'
    ];

    return $feriados;
}

/**
 * Calcula las fechas de Semana Santa para un año dado
 * Algoritmo basado en el cálculo eclesiástico
 *
 * @param int $anio Año a calcular
 * @return array Array con jueves_santo y viernes_santo en formato YYYY-MM-DD
 */
function calcularSemanaSanta($anio) {
    // Algoritmo de Gauss para calcular el Domingo de Pascua
    $a = $anio % 19;
    $b = floor($anio / 100);
    $c = $anio % 100;
    $d = floor($b / 4);
    $e = $b % 4;
    $f = floor(($b + 8) / 25);
    $g = floor(($b - $f + 1) / 3);
    $h = (19 * $a + $b - $d - $g + 15) % 30;
    $i = floor($c / 4);
    $k = $c % 4;
    $l = (32 + 2 * $e + 2 * $i - $h - $k) % 7;
    $m = floor(($a + 11 * $h + 22 * $l) / 451);
    $mes = floor(($h + $l - 7 * $m + 114) / 31);
    $dia = (($h + $l - 7 * $m + 114) % 31) + 1;

    // Domingo de Pascua
    $pascua = mktime(0, 0, 0, $mes, $dia, $anio);

    // Jueves Santo = Pascua - 3 días
    $jueves_santo = date('Y-m-d', strtotime('-3 days', $pascua));

    // Viernes Santo = Pascua - 2 días
    $viernes_santo = date('Y-m-d', strtotime('-2 days', $pascua));

    return [
        'jueves_santo' => $jueves_santo,
        'viernes_santo' => $viernes_santo
    ];
}
?>
