<?php
/**
 * Funciones auxiliares del sistema SICA
 */

// Iniciar sesión si no está iniciada
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

// Verificar si el usuario está autenticado
function estaAutenticado() {
    return isset($_SESSION['usuario_id']) && !empty($_SESSION['usuario_id']);
}

// Verificar si el usuario tiene un rol específico
function tieneRol($roles) {
    if (!estaAutenticado()) {
        return false;
    }
    if (!is_array($roles)) {
        $roles = [$roles];
    }
    return in_array($_SESSION['usuario_rol'], $roles);
}

// Redirigir si no está autenticado
function requerirAutenticacion() {
    if (!estaAutenticado()) {
        header('Location: /SICA/login.php');
        exit;
    }
}

// Redirigir si no tiene el rol requerido
function requerirRol($roles) {
    requerirAutenticacion();
    if (!tieneRol($roles)) {
        header('Location: /SICA/acceso_denegado.php');
        exit;
    }
}

// Sanitar entrada de datos
function sanitizar($dato) {
    return htmlspecialchars(trim($dato), ENT_QUOTES, 'UTF-8');
}

// Generar token CSRF
function generarTokenCSRF() {
    if (!isset($_SESSION['csrf_token'])) {
        $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
    }
    return $_SESSION['csrf_token'];
}

// Verificar token CSRF
function verificarTokenCSRF($token) {
    return isset($_SESSION['csrf_token']) && hash_equals($_SESSION['csrf_token'], $token);
}

// Formato de fecha
function formatearFecha($fecha, $formato = 'd/m/Y') {
    $fecha_obj = DateTime::createFromFormat('Y-m-d', $fecha);
    return $fecha_obj ? $fecha_obj->format($formato) : $fecha;
}

// Formato de datetime
function formatearDateTime($datetime, $formato = 'd/m/Y H:i') {
    $fecha_obj = DateTime::createFromFormat('Y-m-d H:i:s', $datetime);
    return $fecha_obj ? $fecha_obj->format($formato) : $datetime;
}

// Obtener nombre del rol
function nombreRol($rol) {
    $roles = [
        'admin' => 'Administrador',
        'docente' => 'Docente',
        'auxiliar' => 'Auxiliar',
        'estudiante' => 'Estudiante',
        'apoderado' => 'Apoderado'
    ];
    return $roles[$rol] ?? ucfirst($rol);
}

// Alertas para mostrar en la vista
function setAlerta($tipo, $mensaje) {
    $_SESSION['alerta'] = ['tipo' => $tipo, 'mensaje' => $mensaje];
}

function getAlerta() {
    if (isset($_SESSION['alerta'])) {
        $alerta = $_SESSION['alerta'];
        unset($_SESSION['alerta']);
        return $alerta;
    }
    return null;
}

// Paginación
function paginar($total_registros, $por_pagina = 10, $pagina_actual = 1) {
    $total_paginas = ceil($total_registros / $por_pagina);
    $pagina_actual = max(1, min($pagina_actual, $total_paginas));
    $offset = ($pagina_actual - 1) * $por_pagina;

    return [
        'total' => $total_registros,
        'por_pagina' => $por_pagina,
        'pagina_actual' => $pagina_actual,
        'total_paginas' => $total_paginas,
        'offset' => $offset,
        'prev' => $pagina_actual > 1 ? $pagina_actual - 1 : null,
        'next' => $pagina_actual < $total_paginas ? $pagina_actual + 1 : null
    ];
}

// Generar paginación HTML
function htmlPaginacion($paginacion, $base_url) {
    if ($paginacion['total_paginas'] <= 1) {
        return '';
    }

    ob_start();
    ?>
    <nav aria-label="Paginación">
        <ul class="pagination justify-content-center">
            <li class="page-item <?= $paginacion['prev'] ? '' : 'disabled' ?>">
                <a class="page-link" href="<?= $base_url ?>?pagina=<?= $paginacion['prev'] ?? $paginacion['pagina_actual'] ?>">Anterior</a>
            </li>
            <?php for ($i = 1; $i <= $paginacion['total_paginas']; $i++): ?>
                <li class="page-item <?= $i == $paginacion['pagina_actual'] ? 'active' : '' ?>">
                    <a class="page-link" href="<?= $base_url ?>?pagina=<?= $i ?>"><?= $i ?></a>
                </li>
            <?php endfor; ?>
            <li class="page-item <?= $paginacion['next'] ? '' : 'disabled' ?>">
                <a class="page-link" href="<?= $base_url ?>?pagina=<?= $paginacion['next'] ?? $paginacion['pagina_actual'] ?>">Siguiente</a>
            </li>
        </ul>
    </nav>
    <?php
    return ob_get_clean();
}

// ============================================
// GESTIÓN DE AÑO LECTIVO EN SESIÓN
// ============================================

// Obtener el año lectivo seleccionado en sesión
function getAnioLectivoSeleccionado() {
    global $pdo;

    // Si ya hay un año seleccionado en sesión, retornarlo
    if (isset($_SESSION['anio_lectivo_id'])) {
        return $_SESSION['anio_lectivo_id'];
    }

    // Si no, buscar el año activo y guardarlo en sesión
    $anio_activo = query("SELECT id FROM anios_lectivos WHERE estado = 'activo' LIMIT 1")->fetch();

    if ($anio_activo) {
        $_SESSION['anio_lectivo_id'] = $anio_activo['id'];
        return $anio_activo['id'];
    }

    // Si no hay ningún año, retornar null
    return null;
}

// Establecer el año lectivo seleccionado
function setAnioLectivoSeleccionado($anio_id) {
    global $pdo;

    // Verificar que el año existe
    $anio = query("SELECT id, anio, estado FROM anios_lectivos WHERE id = ?", [$anio_id])->fetch();

    if ($anio) {
        $_SESSION['anio_lectivo_id'] = $anio_id;
        $_SESSION['anio_lectivo_nombre'] = $anio['anio'];
        return true;
    }

    return false;
}

// Obtener información del año lectivo seleccionado
function getInfoAnioLectivoSeleccionado() {
    global $pdo;
    $anio_id = getAnioLectivoSeleccionado();

    if (!$anio_id) {
        return null;
    }

    return query("SELECT * FROM anios_lectivos WHERE id = ?", [$anio_id])->fetch();
}

// Respuesta JSON para endpoints AJAX
function json_response($data, $status_code = 200) {
    http_response_code($status_code);
    header('Content-Type: application/json');
    echo json_encode($data);
    exit;
}
