<?php
require_once '../config/db.php';
require_once '../config/functions.php';

requerirRol(['admin']);

// Acción: Crear nuevo estado
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'crear') {
    $nombre = sanitizar($_POST['nombre'] ?? '');
    $abreviatura = sanitizar($_POST['abreviatura'] ?? '');
    $descripcion = sanitizar($_POST['descripcion'] ?? '');
    $color = sanitizar($_POST['color'] ?? '#6c757d');
    $icono = sanitizar($_POST['icono'] ?? 'circle-fill');
    $conteo_asistencia = sanitizar($_POST['conteo_asistencia'] ?? 'si');
    $permite_tardanza = isset($_POST['permite_tardanza']) ? 1 : 0;
    $requiere_justificacion = isset($_POST['requiere_justificacion']) ? 1 : 0;

    // Validaciones
    if (empty($nombre) || empty($abreviatura)) {
        $error = 'El nombre y la abreviatura son obligatorios';
    } else {
        // Verificar duplicados
        $existe = query("SELECT id FROM estados_asistencia WHERE nombre = ? OR abreviatura = ?", [$nombre, $abreviatura])->fetch();

        if ($existe) {
            $error = 'Ya existe un estado con ese nombre o abreviatura';
        } else {
            $sql = "INSERT INTO estados_asistencia (nombre, abreviatura, descripcion, color, icono, conteo_asistencia, permite_tardanza, requiere_justificacion)
                    VALUES (:nombre, :abreviatura, :descripcion, :color, :icono, :conteo_asistencia, :permite_tardanza, :requiere_justificacion)";

            $params = [
                'nombre' => $nombre,
                'abreviatura' => $abreviatura,
                'descripcion' => $descripcion,
                'color' => $color,
                'icono' => $icono,
                'conteo_asistencia' => $conteo_asistencia,
                'permite_tardanza' => $permite_tardanza,
                'requiere_justificacion' => $requiere_justificacion
            ];

            if (query($sql, $params)) {
                $exito = 'Estado de asistencia creado correctamente';
            } else {
                $error = 'Error al crear el estado de asistencia';
            }
        }
    }
}

// Acción: Editar estado
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'editar') {
    $id = (int)($_POST['id'] ?? 0);
    $nombre = sanitizar($_POST['nombre'] ?? '');
    $abreviatura = sanitizar($_POST['abreviatura'] ?? '');
    $descripcion = sanitizar($_POST['descripcion'] ?? '');
    $color = sanitizar($_POST['color'] ?? '#6c757d');
    $icono = sanitizar($_POST['icono'] ?? 'circle-fill');
    $conteo_asistencia = sanitizar($_POST['conteo_asistencia'] ?? 'si');
    $permite_tardanza = isset($_POST['permite_tardanza']) ? 1 : 0;
    $requiere_justificacion = isset($_POST['requiere_justificacion']) ? 1 : 0;
    $estado = sanitizar($_POST['estado'] ?? 'activo');

    // Validaciones
    if (empty($nombre) || empty($abreviatura)) {
        $error = 'El nombre y la abreviatura son obligatorios';
    } else {
        // Verificar duplicados (excluyendo el registro actual)
        $existe = query("SELECT id FROM estados_asistencia WHERE (nombre = ? OR abreviatura = ?) AND id != ?", [$nombre, $abreviatura, $id])->fetch();

        if ($existe) {
            $error = 'Ya existe otro estado con ese nombre o abreviatura';
        } else {
            $sql = "UPDATE estados_asistencia
                    SET nombre = :nombre, abreviatura = :abreviatura, descripcion = :descripcion,
                        color = :color, icono = :icono, conteo_asistencia = :conteo_asistencia,
                        permite_tardanza = :permite_tardanza, requiere_justificacion = :requiere_justificacion,
                        estado = :estado
                    WHERE id = :id";

            $params = [
                'nombre' => $nombre,
                'abreviatura' => $abreviatura,
                'descripcion' => $descripcion,
                'color' => $color,
                'icono' => $icono,
                'conteo_asistencia' => $conteo_asistencia,
                'permite_tardanza' => $permite_tardanza,
                'requiere_justificacion' => $requiere_justificacion,
                'estado' => $estado,
                'id' => $id
            ];

            if (query($sql, $params)) {
                $exito = 'Estado de asistencia actualizado correctamente';
            } else {
                $error = 'Error al actualizar el estado de asistencia';
            }
        }
    }
}

// Acción: Activar/Inactivar
if (isset($_GET['activar'])) {
    $id = (int)$_GET['activar'];
    $nuevo_estado = $_GET['estado'] ?? 'activo';
    query("UPDATE estados_asistencia SET estado = ? WHERE id = ?", [$nuevo_estado, $id]);
    header("Location: estados_asistencia.php");
    exit;
}

// Acción: Eliminar estado
if (isset($_GET['eliminar'])) {
    $id = (int)$_GET['eliminar'];

    try {
        // Verificar si se está usando en asistencias
        $en_uso = query("SELECT COUNT(*) as total FROM asistencias WHERE id_estado_asistencia = ?", [$id])->fetch();

        if ($en_uso && $en_uso['total'] > 0) {
            $error = "No se puede eliminar el estado porque está siendo usado en {$en_uso['total']} registro(s) de asistencia.";
        } else {
            // Eliminar definitivamente
            query("DELETE FROM estados_asistencia WHERE id = ?", [$id]);
            $exito = 'Estado de asistencia eliminado correctamente';
        }
    } catch (Exception $e) {
        // Si hay error (probablemente la tabla asistencias no tiene el campo aún), permitir eliminar
        query("DELETE FROM estados_asistencia WHERE id = ?", [$id]);
        $exito = 'Estado de asistencia eliminado correctamente';
    }
}

// Obtener estados de asistencia
try {
    $estados = query("SELECT * FROM estados_asistencia ORDER BY id")->fetchAll();
} catch (Exception $e) {
    // Si la tabla no existe, mostrar array vacío
    $estados = [];
}

// Estadísticas
$total = count($estados);
$activos = count(array_filter($estados, fn($e) => $e['estado'] === 'activo'));
$inactivos = $total - $activos;

?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0, maximum-scale=1.0, user-scalable=no">
    <title>SICA - Estados de Asistencia</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.1/font/bootstrap-icons.css" rel="stylesheet">
    <link href="../assets/css/sica.css?v=5.1" rel="stylesheet">
</head>
<body>
    <?php include '../includes/layout_elems.php'; ?>

    <div class="d-flex justify-content-between align-items-center mb-4">
        <div class="d-flex align-items-center">
            <button class="btn btn-link d-lg-none me-3" onclick="toggleSidebarMobile()">
                <i class="bi bi-list fs-4"></i>
            </button>
            <div>
                <h3 class="mb-1">Estados de Asistencia</h3>
                <p class="text-muted mb-0">Gestione los estados para registros de asistencia</p>
            </div>
        </div>
        <button class="btn btn-primary" onclick="mostrarFormulario()">
            <i class="bi bi-plus-circle-fill me-2"></i>Nuevo Estado
        </button>
    </div>

    <!-- Alertas -->
    <?php if (isset($error)): ?>
        <div class="alert alert-danger alert-dismissible fade show" role="alert">
            <i class="bi bi-exclamation-triangle-fill me-2"></i><?= htmlspecialchars($error) ?>
            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
        </div>
    <?php endif; ?>

    <?php if (isset($exito)): ?>
        <div class="alert alert-success alert-dismissible fade show" role="alert">
            <i class="bi bi-check-circle-fill me-2"></i><?= htmlspecialchars($exito) ?>
            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
        </div>
    <?php endif; ?>

    <!-- Estadísticas -->
    <div class="row mb-4">
        <div class="col-md-4">
            <div class="card border-0 shadow-sm">
                <div class="card-body">
                    <div class="d-flex align-items-center">
                        <div class="me-3">
                            <div class="bg-primary bg-opacity-10 p-3 rounded">
                                <i class="bi bi-list-check fs-4 text-primary"></i>
                            </div>
                        </div>
                        <div>
                            <h6 class="mb-0 text-muted">Total</h6>
                            <h3 class="mb-0"><?= $total ?></h3>
                        </div>
                    </div>
                </div>
            </div>
        </div>
        <div class="col-md-4">
            <div class="card border-0 shadow-sm">
                <div class="card-body">
                    <div class="d-flex align-items-center">
                        <div class="me-3">
                            <div class="bg-success bg-opacity-10 p-3 rounded">
                                <i class="bi bi-check-circle-fill fs-4 text-success"></i>
                            </div>
                        </div>
                        <div>
                            <h6 class="mb-0 text-muted">Activos</h6>
                            <h3 class="mb-0"><?= $activos ?></h3>
                        </div>
                    </div>
                </div>
            </div>
        </div>
        <div class="col-md-4">
            <div class="card border-0 shadow-sm">
                <div class="card-body">
                    <div class="d-flex align-items-center">
                        <div class="me-3">
                            <div class="bg-secondary bg-opacity-10 p-3 rounded">
                                <i class="bi bi-dash-circle-fill fs-4 text-secondary"></i>
                            </div>
                        </div>
                        <div>
                            <h6 class="mb-0 text-muted">Inactivos</h6>
                            <h3 class="mb-0"><?= $inactivos ?></h3>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Formulario (oculto por defecto) -->
    <div id="formularioCard" class="card border-0 shadow-sm mb-4" style="display: none;">
        <div class="card-header bg-white">
            <h5 class="mb-0" id="formTitulo">
                <i class="bi bi-plus-circle-fill me-2"></i>Nuevo Estado de Asistencia
            </h5>
        </div>
        <div class="card-body">
            <form id="formularioEstado" method="POST">
                <input type="hidden" name="action" id="action" value="crear">
                <input type="hidden" name="id" id="id">

                <div class="row">
                    <div class="col-md-6 mb-3">
                        <label class="form-label">Nombre <span class="text-danger">*</span></label>
                        <input type="text" class="form-control" name="nombre" id="nombre" required>
                    </div>
                    <div class="col-md-3 mb-3">
                        <label class="form-label">Abreviatura <span class="text-danger">*</span></label>
                        <input type="text" class="form-control" name="abreviatura" id="abreviatura" maxlength="10" required>
                    </div>
                    <div class="col-md-3 mb-3">
                        <label class="form-label">Color</label>
                        <div class="input-group">
                            <input type="color" class="form-control form-control-color" name="color" id="color" value="#6c757d">
                            <input type="text" class="form-control" name="color_text" id="color_text" value="#6c757d">
                        </div>
                    </div>
                </div>

                <div class="mb-3">
                    <label class="form-label">Descripción</label>
                    <textarea class="form-control" name="descripcion" id="descripcion" rows="2"></textarea>
                </div>

                <div class="row">
                    <div class="col-md-4 mb-3">
                        <label class="form-label">Icono (Bootstrap Icons)</label>
                        <input type="text" class="form-control" name="icono" id="icono" value="circle-fill">
                        <div class="form-text">Ej: check-circle-fill, clock-fill, x-circle-fill</div>
                    </div>
                    <div class="col-md-4 mb-3">
                        <label class="form-label">Cuenta como Asistencia</label>
                        <select class="form-select" name="conteo_asistencia" id="conteo_asistencia">
                            <option value="si">Sí</option>
                            <option value="no">No</option>
                        </select>
                    </div>
                    <div class="col-md-4 mb-3" id="campoEstado">
                        <label class="form-label">Estado</label>
                        <select class="form-select" name="estado" id="selectEstado">
                            <option value="activo">Activo</option>
                            <option value="inactivo">Inactivo</option>
                        </select>
                    </div>
                </div>

                <div class="row">
                    <div class="col-md-6 mb-3">
                        <div class="form-check">
                            <input class="form-check-input" type="checkbox" name="permite_tardanza" id="permite_tardanza">
                            <label class="form-check-label" for="permite_tardanza">
                                Permite registrar minutos de tardanza
                            </label>
                        </div>
                    </div>
                    <div class="col-md-6 mb-3">
                        <div class="form-check">
                            <input class="form-check-input" type="checkbox" name="requiere_justificacion" id="requiere_justificacion">
                            <label class="form-check-label" for="requiere_justificacion">
                                Requiere justificación/documento
                            </label>
                        </div>
                    </div>
                </div>

                <div class="d-flex gap-2">
                    <button type="submit" class="btn btn-primary">
                        <i class="bi bi-save-fill me-2"></i>Guardar
                    </button>
                    <button type="button" class="btn btn-secondary" onclick="ocultarFormulario()">
                        Cancelar
                    </button>
                </div>
            </form>
        </div>
    </div>

    <!-- Tabla de estados -->
    <div class="card border-0 shadow-sm">
        <div class="card-body">
            <div class="table-responsive">
                <table class="table table-hover">
                    <thead>
                        <tr>
                            <th>Color</th>
                            <th>Abreviatura</th>
                            <th>Nombre</th>
                            <th>Descripción</th>
                            <th>¿Cuenta?</th>
                            <th>Tardanza</th>
                            <th>Justificación</th>
                            <th>Estado</th>
                            <th class="text-end">Acciones</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($estados as $estado): ?>
                            <tr>
                                <td>
                                    <span class="badge d-inline-flex align-items-center" style="background-color: <?= htmlspecialchars($estado['color']) ?>;">
                                        <i class="bi bi-<?= htmlspecialchars($estado['icono']) ?> me-1"></i>
                                        <?= htmlspecialchars($estado['abreviatura']) ?>
                                    </span>
                                </td>
                                <td><strong><?= htmlspecialchars($estado['abreviatura']) ?></strong></td>
                                <td><?= htmlspecialchars($estado['nombre']) ?></td>
                                <td class="text-muted small"><?= htmlspecialchars($estado['descripcion'] ?: '-') ?></td>
                                <td>
                                    <?php if ($estado['conteo_asistencia'] === 'si'): ?>
                                        <span class="badge bg-success">Sí</span>
                                    <?php else: ?>
                                        <span class="badge bg-danger">No</span>
                                    <?php endif; ?>
                                </td>
                                <td>
                                    <?php if ($estado['permite_tardanza']): ?>
                                        <span class="badge bg-info">Sí</span>
                                    <?php else: ?>
                                        <span class="text-muted">-</span>
                                    <?php endif; ?>
                                </td>
                                <td>
                                    <?php if ($estado['requiere_justificacion']): ?>
                                        <span class="badge bg-warning text-dark">Sí</span>
                                    <?php else: ?>
                                        <span class="text-muted">-</span>
                                    <?php endif; ?>
                                </td>
                                <td>
                                    <?php if ($estado['estado'] === 'activo'): ?>
                                        <span class="badge bg-success">Activo</span>
                                    <?php else: ?>
                                        <span class="badge bg-secondary">Inactivo</span>
                                    <?php endif; ?>
                                </td>
                                <td class="text-end">
                                    <button class="btn btn-sm btn-outline-primary me-1" onclick="editarEstado(<?= htmlspecialchars(json_encode($estado)) ?>)">
                                        <i class="bi bi-pencil-fill"></i>
                                    </button>
                                    <?php if ($estado['estado'] === 'activo'): ?>
                                        <a href="?activar=<?= $estado['id'] ?>&estado=inactivo" class="btn btn-sm btn-outline-warning me-1" title="Inactivar">
                                            <i class="bi bi-pause-fill"></i>
                                        </a>
                                    <?php else: ?>
                                        <a href="?activar=<?= $estado['id'] ?>&estado=activo" class="btn btn-sm btn-outline-success me-1" title="Activar">
                                            <i class="bi bi-play-fill"></i>
                                        </a>
                                    <?php endif; ?>
                                    <button class="btn btn-sm btn-outline-danger" onclick="confirmarEliminar(<?= $estado['id'] ?>, '<?= htmlspecialchars($estado['nombre']) ?>')" title="Eliminar">
                                        <i class="bi bi-trash-fill"></i>
                                    </button>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        </div>
    </div>

    <?php include '../includes/layout_scripts.php'; ?>

    <script>
        // Sincronizar color
        document.getElementById('color').addEventListener('input', function() {
            document.getElementById('color_text').value = this.value;
        });
        document.getElementById('color_text').addEventListener('input', function() {
            document.getElementById('color').value = this.value;
        });

        function mostrarFormulario() {
            document.getElementById('formularioCard').style.display = 'block';
            document.getElementById('formTitulo').innerHTML = '<i class="bi bi-plus-circle-fill me-2"></i>Nuevo Estado de Asistencia';
            document.getElementById('action').value = 'crear';
            document.getElementById('formularioEstado').reset();
            document.getElementById('color').value = '#6c757d';
            document.getElementById('color_text').value = '#6c757d';
            document.getElementById('campoEstado').style.display = 'none';
            document.getElementById('nombre').focus();
        }

        function ocultarFormulario() {
            document.getElementById('formularioCard').style.display = 'none';
        }

        function editarEstado(estado) {
            document.getElementById('formularioCard').style.display = 'block';
            document.getElementById('formTitulo').innerHTML = '<i class="bi bi-pencil-fill me-2"></i>Editar Estado de Asistencia';
            document.getElementById('action').value = 'editar';
            document.getElementById('id').value = estado.id;
            document.getElementById('nombre').value = estado.nombre;
            document.getElementById('abreviatura').value = estado.abreviatura;
            document.getElementById('descripcion').value = estado.descripcion || '';
            document.getElementById('color').value = estado.color;
            document.getElementById('color_text').value = estado.color;
            document.getElementById('icono').value = estado.icono;
            document.getElementById('conteo_asistencia').value = estado.conteo_asistencia;
            document.getElementById('permite_tardanza').checked = estado.permite_tardanza == 1;
            document.getElementById('requiere_justificacion').checked = estado.requiere_justificacion == 1;
            document.getElementById('selectEstado').value = estado.estado;
            document.getElementById('campoEstado').style.display = 'block';
            document.getElementById('nombre').focus();

            // Scroll hacia el formulario
            document.getElementById('formularioCard').scrollIntoView({ behavior: 'smooth' });
        }

        function confirmarEliminar(id, nombre) {
            if (confirm(`¿Está seguro de eliminar el estado "${nombre}"?\n\nEsta acción eliminará definitivamente el estado y no se puede deshacer.`)) {
                window.location.href = `?eliminar=${id}`;
            }
        }
    </script>
</body>
</html>
