<?php
require_once '../config/db.php';
require_once '../config/functions.php';

requerirRol(['admin']);

// Función auxiliar para traducir días de la semana al español
function diaSemanaEspanol($fecha) {
    $dias = [
        'Monday' => 'Lunes',
        'Tuesday' => 'Martes',
        'Wednesday' => 'Miércoles',
        'Thursday' => 'Jueves',
        'Friday' => 'Viernes',
        'Saturday' => 'Sábado',
        'Sunday' => 'Domingo'
    ];
    $diaIngles = date('l', strtotime($fecha));
    return $dias[$diaIngles] ?? $diaIngles;
}

$anio_activo = getAnioLectivoSeleccionado();

$exito = null;
$error = null;

// Acción: Guardar (crear o editar)
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $fecha = sanitizar($_POST['fecha'] ?? '');
    $tipo = sanitizar($_POST['tipo'] ?? '');
    $nombre = sanitizar($_POST['nombre'] ?? '');
    $descripcion = sanitizar($_POST['descripcion'] ?? '');
    $afecta_todos_niveles = isset($_POST['afecta_todos_niveles']) ? 1 : 0;
    $niveles_afectados = $_POST['niveles_afectados'] ?? [];
    $observaciones = sanitizar($_POST['observaciones'] ?? '');
    $estado = sanitizar($_POST['estado'] ?? 'activo');
    $editando_id = isset($_POST['editando_id']) ? (int)$_POST['editando_id'] : null;

    // Validaciones
    if (empty($fecha)) {
        $error = 'La fecha es obligatoria';
    } elseif (empty($tipo)) {
        $error = 'El tipo es obligatorio';
    } elseif (empty($nombre)) {
        $error = 'El nombre es obligatorio';
    } else {
        // Convertir array de niveles a JSON
        $niveles_json = !empty($niveles_afectados) ? json_encode($niveles_afectados) : null;

        // Usar siempre el año lectivo activo
        $aniolectivo_id = $anio_activo;

        if ($editando_id) {
            // Verificar duplicado de fecha (excluyendo el registro actual)
            $duplicado = query("SELECT id FROM calendario WHERE fecha = ? AND id != ?", [$fecha, $editando_id])->fetch();
            if ($duplicado) {
                $error = 'Ya existe un registro para esta fecha';
            } else {
                // Actualizar
                $sql = "UPDATE calendario SET
                        fecha = ?,
                        tipo = ?,
                        nombre = ?,
                        descripcion = ?,
                        aniolectivo_id = ?,
                        afecta_todos_niveles = ?,
                        niveles_afectados = ?,
                        observaciones = ?,
                        estado = ?
                        WHERE id = ?";

                query($sql, [
                    $fecha,
                    $tipo,
                    $nombre,
                    $descripcion,
                    $aniolectivo_id,  // ← Usar variable calculada (NULL o ID)
                    $afecta_todos_niveles,
                    $niveles_json,
                    $observaciones,
                    $estado,
                    $editando_id
                ]);

                $exito = 'Calendario actualizado correctamente';
                $editando_id = null;
            }
        } else {
            // Verificar duplicado de fecha
            $duplicado = query("SELECT id FROM calendario WHERE fecha = ?", [$fecha])->fetch();
            if ($duplicado) {
                $error = 'Ya existe un registro para esta fecha';
            } else {
                // Crear
                $sql = "INSERT INTO calendario
                        (fecha, tipo, nombre, descripcion, aniolectivo_id, afecta_todos_niveles, niveles_afectados, observaciones, estado)
                        VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)";

                query($sql, [
                    $fecha,
                    $tipo,
                    $nombre,
                    $descripcion,
                    $aniolectivo_id,  // ← Usar variable calculada (NULL o ID)
                    $afecta_todos_niveles,
                    $niveles_json,
                    $observaciones,
                    $estado
                ]);

                $exito = 'Calendario registrado correctamente';
            }
        }
    }
}

// Acción: Editar
$editando = null;
if (isset($_GET['editar'])) {
    $id = (int)$_GET['editar'];
    $editando = query("SELECT * FROM calendario WHERE id = ?", [$id])->fetch();
}

// Acción: Eliminar
if (isset($_GET['eliminar'])) {
    $id = (int)$_GET['eliminar'];

    try {
        // Obtener información del registro
        $registro = query("SELECT fecha FROM calendario WHERE id = ?", [$id])->fetch();

        if ($registro) {
            $fecha_pasada = $registro['fecha'] < date('Y-m-d');

            // No eliminar registros de fecha pasada
            if ($fecha_pasada) {
                $error = 'No se puede eliminar un registro de fecha pasada. Puede desactivarlo en su lugar.';
            } else {
                query("DELETE FROM calendario WHERE id = ?", [$id]);
                $exito = 'Registro eliminado correctamente';
            }
        }
    } catch (Exception $e) {
        $error = 'Error al eliminar el registro';
    }
}

// Acción: Activar/Desactivar
if (isset($_GET['toggle'])) {
    $id = (int)$_GET['toggle'];
    $actual = query("SELECT estado FROM calendario WHERE id = ?", [$id])->fetch();

    if ($actual) {
        $nuevo_estado = $actual['estado'] === 'activo' ? 'inactivo' : 'activo';
        query("UPDATE calendario SET estado = ? WHERE id = ?", [$nuevo_estado, $id]);
        $exito = 'Estado actualizado correctamente';
    }
}

// Obtener registros del calendario
// IMPORTANTE:
// - Mostrar registros ESPECÍFICOS del año lectivo activo
// - O mostrar registros cuyas fechas coincidan con el año del año lectivo activo
try {
    // Obtener el año del año lectivo activo (extraer el año del nombre "2025" o "2026")
    $anio_lectivo_info = query("SELECT anio FROM anios_lectivos WHERE id = ?", [$anio_activo])->fetch();
    $anio_filtro = $anio_lectivo_info ? (int)substr($anio_lectivo_info['anio'], 0, 4) : (int)date('Y');

    $calendario = query("
        SELECT *
        FROM calendario
        WHERE estado = 'activo'
          AND (
              aniolectivo_id = ?  -- Específicos del año lectivo activo
              OR YEAR(fecha) = ?  -- O registros del año correspondiente
          )
        ORDER BY fecha ASC
    ", [$anio_activo, $anio_filtro])->fetchAll();
} catch (Exception $e) {
    $calendario = [];
}

// Obtener niveles para el formulario
$niveles = query("SELECT * FROM niveles WHERE estado = 'activo' ORDER BY nombre")->fetchAll();
?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0, maximum-scale=1.0, user-scalable=no">
    <title>SICA - Gestión de Calendario</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.1/font/bootstrap-icons.css" rel="stylesheet">
    <link href="../assets/css/sica.css?v=5.1" rel="stylesheet">
    <style>
        .tipo-badge {
            font-size: 0.85rem;
            padding: 0.4rem 0.8rem;
            border-radius: 8px;
            font-weight: 500;
        }
        .tipo-feriado { background-color: #fee2e2; color: #dc2626; border: 1px solid #fca5a5; }
        .tipo-vacacion { background-color: #dbeafe; color: #2563eb; border: 1px solid #93c5fd; }
        .tipo-dia_no_laboral { background-color: #fef3c7; color: #d97706; border: 1px solid #fcd34d; }
        .tipo-fin_semana_largo { background-color: #e0e7ff; color: #4f46e5; border: 1px solid #a5b4fc; }

        .fecha-cell {
            font-weight: 600;
            color: #374151;
        }

        .nivel-tag {
            font-size: 0.75rem;
            padding: 0.2rem 0.5rem;
            border-radius: 4px;
            background-color: #f3f4f6;
            color: #4b5563;
            margin-right: 0.25rem;
        }
    </style>
</head>
<body>
    <?php include '../includes/layout_elems.php'; ?>

    <div class="d-flex justify-content-between align-items-center mb-4">
        <div class="d-flex align-items-center">
            <button class="btn btn-link d-lg-none me-3" onclick="toggleSidebarMobile()">
                <i class="bi bi-list fs-4"></i>
            </button>
            <div>
                <h3 class="mb-1">Gestión de Calendario</h3>
                <p class="text-muted mb-0">Feriados, vacaciones y días no laborables - <?= $anio_activo ?></p>
            </div>
        </div>
        <div>
            <a href="../dashboard.php" class="btn btn-outline-primary">
                <i class="bi bi-house-fill me-2"></i>Dashboard
            </a>
        </div>
    </div>

    <div class="row">
        <!-- Columna Izquierda: Formulario -->
        <div class="col-lg-4 mb-4">
            <div class="card shadow-sm">
                <div class="card-header bg-primary text-white">
                    <h5 class="mb-0"><i class="bi bi-calendar-plus me-2"></i><?= $editando ? 'Editar Registro' : 'Nuevo Registro' ?></h5>
                </div>
                <div class="card-body">
                    <?php if ($exito): ?>
                        <div class="alert alert-success alert-dismissible fade show" role="alert">
                            <i class="bi bi-check-circle-fill me-2"></i><?= htmlspecialchars($exito) ?>
                            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                        </div>
                    <?php endif; ?>

                    <?php if ($error): ?>
                        <div class="alert alert-danger alert-dismissible fade show" role="alert">
                            <i class="bi bi-exclamation-triangle-fill me-2"></i><?= htmlspecialchars($error) ?>
                            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                        </div>
                    <?php endif; ?>

                    <form method="POST" id="formularioCalendario">
                        <input type="hidden" name="editando_id" value="<?= $editando['id'] ?? '' ?>">

                        <div class="mb-3">
                            <label class="form-label fw-bold">Fecha *</label>
                            <input type="date"
                                   class="form-control"
                                   name="fecha"
                                   value="<?= $editando['fecha'] ?? '' ?>"
                                   required>
                        </div>

                        <div class="mb-3">
                            <label class="form-label fw-bold">Tipo *</label>
                            <select class="form-select" name="tipo" id="tipoSelect" required onchange="actualizarNombre()">
                                <option value="">Seleccione...</option>
                                <option value="feriado" <?= ($editando['tipo'] ?? '') === 'feriado' ? 'selected' : '' ?>>Feriado</option>
                                <option value="vacacion" <?= ($editando['tipo'] ?? '') === 'vacacion' ? 'selected' : '' ?>>Vacación</option>
                                <option value="dia_no_laboral" <?= ($editando['tipo'] ?? '') === 'dia_no_laboral' ? 'selected' : '' ?>>Día No Laboral</option>
                                <option value="fin_semana_largo" <?= ($editando['tipo'] ?? '') === 'fin_semana_largo' ? 'selected' : '' ?>>Fin de Semana Largo</option>
                            </select>
                        </div>

                        <div class="mb-3">
                            <label class="form-label fw-bold">Nombre *</label>
                            <input type="text"
                                   class="form-control"
                                   name="nombre"
                                   id="nombreInput"
                                   value="<?= htmlspecialchars($editando['nombre'] ?? '') ?>"
                                   placeholder="Ej: Año Nuevo"
                                   required>
                            <div class="form-text">Descripción breve del día festivo o período</div>
                        </div>

                        <div class="mb-3">
                            <label class="form-label fw-bold">Descripción</label>
                            <textarea class="form-control"
                                      name="descripcion"
                                      rows="2"
                                      placeholder="Detalles adicionales (opcional)"><?= htmlspecialchars($editando['descripcion'] ?? '') ?></textarea>
                        </div>

                        <div class="mb-3">
                            <div class="form-check">
                                <input class="form-check-input"
                                       type="checkbox"
                                       name="afecta_todos_niveles"
                                       id="afectaTodosCheck"
                                       <?= empty($editando) || ($editando['afecta_todos_niveles'] ?? 1) ? 'checked' : '' ?>
                                       onchange="toggleNiveles()">
                                <label class="form-check-label fw-bold" for="afectaTodosCheck">
                                    Afecta a todos los niveles
                                </label>
                            </div>
                        </div>

                        <div class="mb-3" id="nivelesContainer" style="display: none;">
                            <label class="form-label fw-bold">Niveles Afectados</label>
                            <div class="border rounded p-3 bg-light">
                                <?php foreach ($niveles as $nivel): ?>
                                    <?php
                                    $seleccionado = false;
                                    if ($editando && !empty($editando['niveles_afectados'])) {
                                        $niveles_arr = json_decode($editando['niveles_afectados'], true);
                                        $seleccionado = in_array($nivel['id'], $niveles_arr ?? []);
                                    }
                                    ?>
                                    <div class="form-check">
                                        <input class="form-check-input"
                                               type="checkbox"
                                               name="niveles_afectados[]"
                                               value="<?= $nivel['id'] ?>"
                                               id="nivel_<?= $nivel['id'] ?>"
                                               <?= $seleccionado ? 'checked' : '' ?>>
                                        <label class="form-check-label" for="nivel_<?= $nivel['id'] ?>">
                                            <?= htmlspecialchars($nivel['nombre']) ?>
                                        </label>
                                    </div>
                                <?php endforeach; ?>
                            </div>
                        </div>

                        <div class="mb-3">
                            <label class="form-label fw-bold">Observaciones</label>
                            <textarea class="form-control"
                                      name="observaciones"
                                      rows="2"
                                      placeholder="Notas o comentarios adicionales"><?= htmlspecialchars($editando['observaciones'] ?? '') ?></textarea>
                        </div>

                        <div class="mb-3">
                            <label class="form-label fw-bold">Estado</label>
                            <select class="form-select" name="estado">
                                <option value="activo" <?= ($editando['estado'] ?? 'activo') === 'activo' ? 'selected' : '' ?>>Activo</option>
                                <option value="inactivo" <?= ($editando['estado'] ?? '') === 'inactivo' ? 'selected' : '' ?>>Inactivo</option>
                            </select>
                        </div>

                        <div class="d-grid gap-2">
                            <button type="submit" class="btn btn-primary">
                                <i class="bi bi-save-fill me-2"></i><?= $editando ? 'Actualizar' : 'Guardar' ?>
                            </button>
                            <?php if ($editando): ?>
                                <a href="calendario.php" class="btn btn-outline-secondary">
                                    <i class="bi bi-x-circle me-2"></i>Cancelar
                                </a>
                            <?php endif; ?>
                        </div>
                    </form>
                </div>
            </div>
        </div>

        <!-- Columna Derecha: Listado -->
        <div class="col-lg-8">
            <div class="card shadow-sm">
                <div class="card-header bg-white">
                    <div class="d-flex justify-content-between align-items-center">
                        <h5 class="mb-0"><i class="bi bi-calendar-check me-2"></i>Registros del Calendario</h5>
                        <span class="badge bg-primary"><?= count($calendario) ?> registros</span>
                    </div>
                </div>
                <div class="card-body p-0">
                    <?php if (empty($calendario)): ?>
                        <div class="text-center py-5">
                            <i class="bi bi-calendar-x text-muted" style="font-size: 3rem;"></i>
                            <p class="text-muted mt-3">No hay registros en el calendario</p>
                        </div>
                    <?php else: ?>
                        <div class="table-responsive">
                            <table class="table table-hover mb-0">
                                <thead class="table-light">
                                    <tr>
                                        <th>Fecha</th>
                                        <th>Tipo</th>
                                        <th>Nombre</th>
                                        <th>Niveles</th>
                                        <th>Estado</th>
                                        <th class="text-center">Acciones</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($calendario as $item): ?>
                                        <?php
                                        $tipo_badge_class = 'tipo-' . $item['tipo'];
                                        $tipo_label = match($item['tipo']) {
                                            'feriado' => 'Feriado',
                                            'vacacion' => 'Vacación',
                                            'dia_no_laboral' => 'Día No Laboral',
                                            'fin_semana_largo' => 'Fin de Semana Largo',
                                            default => $item['tipo']
                                        };
                                        ?>
                                        <tr>
                                            <td class="fecha-cell">
                                                <?= date('d/m/Y', strtotime($item['fecha'])) ?>
                                                <div class="small text-muted">
                                                    <?= diaSemanaEspanol($item['fecha']) ?>
                                                </div>
                                            </td>
                                            <td>
                                                <span class="tipo-badge <?= $tipo_badge_class ?>">
                                                    <?= $tipo_label ?>
                                                </span>
                                            </td>
                                            <td>
                                                <div class="fw-bold"><?= htmlspecialchars($item['nombre']) ?></div>
                                                <?php if (!empty($item['descripcion'])): ?>
                                                    <div class="small text-muted"><?= htmlspecialchars($item['descripcion']) ?></div>
                                                <?php endif; ?>
                                            </td>
                                            <td>
                                                <?php if ($item['afecta_todos_niveles']): ?>
                                                    <span class="badge bg-success">Todos los niveles</span>
                                                <?php elseif (!empty($item['niveles_afectados'])): ?>
                                                    <?php
                                                    $niveles_arr = json_decode($item['niveles_afectados'], true);
                                                    foreach ($niveles_arr as $id_nivel) {
                                                        $nivel_nombre = query("SELECT nombre FROM niveles WHERE id = ?", [$id_nivel])->fetch();
                                                        if ($nivel_nombre) {
                                                            echo '<span class="nivel-tag">' . htmlspecialchars($nivel_nombre['nombre']) . '</span>';
                                                        }
                                                    }
                                                    ?>
                                                <?php else: ?>
                                                    <span class="text-muted small">Sin especificar</span>
                                                <?php endif; ?>
                                            </td>
                                            <td>
                                                <?php if ($item['estado'] === 'activo'): ?>
                                                    <span class="badge bg-success">Activo</span>
                                                <?php else: ?>
                                                    <span class="badge bg-secondary">Inactivo</span>
                                                <?php endif; ?>
                                            </td>
                                            <td class="text-center">
                                                <div class="btn-group" role="group">
                                                    <button class="btn btn-sm btn-outline-primary"
                                                            onclick="window.location.href='?editar=<?= $item['id'] ?>'"
                                                            title="Editar">
                                                        <i class="bi bi-pencil-fill"></i>
                                                    </button>
                                                    <button class="btn btn-sm btn-outline-<?= $item['estado'] === 'activo' ? 'warning' : 'success' ?>"
                                                            onclick="window.location.href='?toggle=<?= $item['id'] ?>'"
                                                            title="<?= $item['estado'] === 'activo' ? 'Desactivar' : 'Activar' ?>">
                                                        <i class="bi bi-<?= $item['estado'] === 'activo' ? 'pause-fill' : 'play-fill' ?>"></i>
                                                    </button>
                                                    <button class="btn btn-sm btn-outline-danger"
                                                            onclick="confirmarEliminar(<?= $item['id'] ?>, '<?= htmlspecialchars($item['nombre']) ?>', '<?= date('d/m/Y', strtotime($item['fecha'])) ?>')"
                                                            title="Eliminar">
                                                        <i class="bi bi-trash-fill"></i>
                                                    </button>
                                                </div>
                                            </td>
                                        </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                    <?php endif; ?>
                </div>
            </div>

            <!-- Panel Informativo -->
            <div class="card shadow-sm mt-4">
                <div class="card-header bg-info text-white">
                    <h6 class="mb-0"><i class="bi bi-info-circle-fill me-2"></i>Información del Sistema</h6>
                </div>
                <div class="card-body">
                    <div class="row">
                        <div class="col-md-6">
                            <h6 class="fw-bold"><i class="bi bi-calendar-event me-2 text-danger"></i>Tipos de Calendario</h6>
                            <ul class="mb-0 small">
                                <li><span class="badge tipo-badge tipo-feriado me-1">Feriado</span> Días festivos nacionales, religiosos, cívicos</li>
                                <li><span class="badge tipo-badge tipo-vacacion me-1">Vacación</span> Periodos de vacaciones escolares</li>
                                <li><span class="badge tipo-badge tipo-dia_no_laboral me-1">Día No Laboral</span> Suspensión de actividades</li>
                                <li><span class="badge tipo-badge tipo-fin_semana_largo me-1">Fin de Semana Largo</span> Fines de semana extendidos</li>
                            </ul>
                            <hr class="my-2">
                            <h6 class="fw-bold mb-2"><i class="bi bi-calendar-range me-2 text-primary"></i>Filtrado por Año</h6>
                            <p class="small text-muted mb-0">
                                El sistema muestra los registros del <strong>año lectivo activo</strong> (<?= $anio_lectivo_info['anio'] ?? $anio_activo ?>).
                                <br>Cambia el año lectivo activo para ver/editar los registros de otros años.
                            </p>
                        </div>
                        <div class="col-md-6">
                            <h6 class="fw-bold"><i class="bi bi-megaphone me-2 text-warning"></i>Impacto en Asistencias</h6>
                            <p class="small text-muted mb-0">
                                Los días registrados en el calendario <strong>NO registrarán asistencia</strong> automáticamente.
                                El sistema mostrará una advertencia si se intenta registrar asistencia en estos días.
                            </p>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <?php include '../includes/layout_scripts.php'; ?>

    <script>
        function toggleNiveles() {
            const afectaTodos = document.getElementById('afectaTodosCheck').checked;
            const nivelesContainer = document.getElementById('nivelesContainer');

            if (afectaTodos) {
                nivelesContainer.style.display = 'none';
            } else {
                nivelesContainer.style.display = 'block';
            }
        }

        function actualizarNombre() {
            const tipo = document.getElementById('tipoSelect').value;
            const fecha = document.querySelector('input[name="fecha"]').value;

            if (tipo && !document.getElementById('nombreInput').value && fecha) {
                // Si el campo está vacío y hay fecha, sugerir nombre según tipo
                const fechaObj = new Date(fecha + 'T00:00:00');
                const opciones = { weekday: 'long', day: 'numeric', month: 'long' };
                const fechaTexto = fechaObj.toLocaleDateString('es-ES', opciones);

                switch(tipo) {
                    case 'feriado':
                        document.getElementById('nombreInput').value = 'Feriado - ' + fechaTexto;
                        break;
                    case 'vacacion':
                        document.getElementById('nombreInput').value = 'Vacación - ' + fechaTexto;
                        break;
                    case 'dia_no_laboral':
                        document.getElementById('nombreInput').value = 'Día No Laboral - ' + fechaTexto;
                        break;
                    case 'fin_semana_largo':
                        document.getElementById('nombreInput').value = 'Fin de Semana Largo - ' + fechaTexto;
                        break;
                }
            }
        }

        function confirmarEliminar(id, nombre, fecha) {
            if (confirm(`¿Está seguro de eliminar "${nombre}" del ${fecha}?\n\nEsta acción no se puede deshacer.`)) {
                window.location.href = `?eliminar=${id}`;
            }
        }

        function espanol(dia) {
            const dias = {
                'Monday': 'Lunes',
                'Tuesday': 'Martes',
                'Wednesday': 'Miércoles',
                'Thursday': 'Jueves',
                'Friday': 'Viernes',
                'Saturday': 'Sábado',
                'Sunday': 'Domingo'
            };
            return dias[dia] || dia;
        }

        // Inicializar
        document.addEventListener('DOMContentLoaded', function() {
            toggleNiveles();
        });
    </script>
</body>
</html>
