<?php
require_once '../config/db.php';
require_once '../config/functions.php';

requerirRol(['admin']);

// Asegurar que solo haya un año activo
$activos = query("SELECT id FROM anios_lectivos WHERE estado = 'activo' ORDER BY id")->fetchAll();
if (count($activos) > 1) {
    // Mantener solo el primero como activo, desactivar los demás
    $primer_id = $activos[0]['id'];
    foreach ($activos as $index => $activo) {
        if ($index > 0) {
            query("UPDATE anios_lectivos SET estado = 'inactivo' WHERE id = ?", [$activo['id']]);
        }
    }
}

$errores = [];
$exito = false;

// Procesar formulario
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $anio = sanitizar($_POST['anio'] ?? '');
    $fecha_inicio = $_POST['fecha_inicio'] ?? '';
    $fecha_fin = $_POST['fecha_fin'] ?? '';
    $accion = $_POST['accion'] ?? '';
    $id_editar = $_POST['id_editar'] ?? '';

    if ($accion === 'crear') {
        if (empty($anio)) $errores[] = 'El año lectivo es obligatorio';
        if (empty($fecha_inicio)) $errores[] = 'La fecha de inicio es obligatoria';
        if (empty($fecha_fin)) $errores[] = 'La fecha de fin es obligatoria';

        $estado_crear = $_POST['estado'] ?? 'inactivo';

        if (empty($errores)) {
            // Verificar si ya existe
            $existe = query("SELECT id FROM anios_lectivos WHERE anio = ?", [$anio])->fetch();
            if ($existe) {
                $errores[] = 'El año lectivo ya existe';
            } else {
                // Si quiere crear como activo, verificar que no haya otro activo
                if ($estado_crear === 'activo') {
                    $activo = query("SELECT COUNT(*) FROM anios_lectivos WHERE estado = 'activo'")->fetchColumn();
                    if ($activo > 0) {
                        $errores[] = 'Ya existe un año lectivo activo. Desactívelo primero o crea este año como inactivo.';
                    }
                }

                if (empty($errores)) {
                    if (query("INSERT INTO anios_lectivos (anio, fecha_inicio, fecha_fin, estado) VALUES (?, ?, ?, ?)",
                        [$anio, $fecha_inicio, $fecha_fin, $estado_crear])) {
                        setAlerta('success', 'Año lectivo creado correctamente');
                    } else {
                        $errores[] = 'Error al crear el año lectivo';
                    }
                }
            }
        }
    } elseif ($accion === 'activar') {
        $id = $_POST['id'] ?? '';

        // Verificar que el año existe
        $anio_info = query("SELECT anio FROM anios_lectivos WHERE id = ?", [$id])->fetch();
        if (!$anio_info) {
            $errores[] = 'El año lectivo no existe';
        } else {
            // Desactivar todos y activar el seleccionado
            query("UPDATE anios_lectivos SET estado = 'inactivo'");
            query("UPDATE anios_lectivos SET estado = 'activo' WHERE id = ?", [$id]);

            // Actualizar sesión
            $_SESSION['anio_lectivo_id'] = $id;
            $_SESSION['anio_lectivo_nombre'] = $anio_info['anio'];

            setAlerta('success', 'Año lectivo activado correctamente');
        }
    } elseif ($accion === 'editar') {
        $id = $_POST['id'] ?? '';
        if (empty($anio)) $errores[] = 'El año lectivo es obligatorio';
        if (empty($fecha_inicio)) $errores[] = 'La fecha de inicio es obligatoria';
        if (empty($fecha_fin)) $errores[] = 'La fecha de fin es obligatoria';

        if (empty($errores)) {
            // Verificar si ya existe otro año con el mismo nombre
            $existe = query("SELECT id FROM anios_lectivos WHERE anio = ? AND id != ?", [$anio, $id])->fetch();
            if ($existe) {
                $errores[] = 'El año lectivo ya existe';
            } else {
                query("UPDATE anios_lectivos SET anio = ?, fecha_inicio = ?, fecha_fin = ? WHERE id = ?",
                    [$anio, $fecha_inicio, $fecha_fin, $id]);
                setAlerta('success', 'Año lectivo actualizado');
            }
        }
    } elseif ($accion === 'eliminar') {
        $id = $_POST['id'] ?? '';

        // Verificar si es el año activo
        $anio_activo_check = query("SELECT estado FROM anios_lectivos WHERE id = ?", [$id])->fetch();
        if ($anio_activo_check && $anio_activo_check['estado'] === 'activo') {
            $errores[] = 'No se puede eliminar el año lectivo activo';
        } else {
            // Verificar si tiene datos relacionados
            // Verificar programaciones
            $tiene_programaciones = query("SELECT COUNT(*) FROM programaciones WHERE id_anio_lectivo = ?", [$id])->fetchColumn();
            if ($tiene_programaciones > 0) {
                $errores[] = 'No se puede eliminar: tiene programaciones asociadas';
            } else {
                query("DELETE FROM anios_lectivos WHERE id = ?", [$id]);
                setAlerta('success', 'Año lectivo eliminado');
            }
        }
    }

    if (empty($errores)) {
        header('Location: anios_lectivos.php');
        exit;
    }
}

// Verificar si estamos editando
$modo_edicion = false;
$anio_editar = null;
if (isset($_GET['editar'])) {
    $id_editar_get = (int)$_GET['editar'];
    $anio_editar = query("SELECT * FROM anios_lectivos WHERE id = ?", [$id_editar_get])->fetch();
    if ($anio_editar) {
        $modo_edicion = true;
    }
}

// Obtener años lectivos
$anios = query("SELECT * FROM anios_lectivos ORDER BY anio DESC")->fetchAll();
$anio_activo = query("SELECT id FROM anios_lectivos WHERE estado = 'activo' LIMIT 1")->fetchColumn();
?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0, maximum-scale=1.0, user-scalable=no">
    <title>SICA - Años Lectivos</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.1/font/bootstrap-icons.css" rel="stylesheet">
    <link href="../assets/css/sica.css?v=5.1" rel="stylesheet">
    <style>
        .content-card { background: white; border-radius: 15px; box-shadow: 0 2px 10px rgba(0,0,0,0.05); padding: 25px; }
        .badge-active { background: #10b981; }
    </style>
</head>
<body>
    <nav class="sidebar"><?php include '../includes/sidebar.php'; ?></nav>

    <div class="main-content">
        <div class="d-flex justify-content-between align-items-center mb-4">
            <div>
                <h3 class="mb-1">Años Lectivos</h3>
                <p class="text-muted mb-0">Gestión de periodos académicos</p>
            </div>
            <a href="index.php" class="btn btn-outline-secondary"><i class="bi bi-arrow-left me-2"></i>Volver</a>
        </div>

        <?php if ($alerta = getAlerta()): ?>
            <div class="alert alert-<?= $alerta['tipo'] ?> alert-dismissible fade show" role="alert">
                <?= $alerta['mensaje'] ?>
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            </div>
        <?php endif; ?>

        <?php if (!empty($errores)): ?>
            <div class="alert alert-danger">
                <?php foreach ($errores as $error): ?><?= $error ?><br><?php endforeach; ?>
            </div>
        <?php endif; ?>

        <div class="row">
            <div class="col-md-4">
                <div class="content-card">
                    <h5 class="mb-3"><?= $modo_edicion ? 'Editar Año Lectivo' : 'Nuevo Año Lectivo' ?></h5>
                    <form method="POST">
                        <input type="hidden" name="accion" value="<?= $modo_edicion ? 'editar' : 'crear' ?>">
                        <?php if ($modo_edicion): ?>
                            <input type="hidden" name="id" value="<?= $anio_editar['id'] ?>">
                        <?php endif; ?>
                        <div class="mb-3">
                            <label class="form-label">Año Lectivo <span class="text-danger">*</span></label>
                            <input type="text" name="anio" class="form-control" placeholder="2025-2026"
                                   value="<?= $modo_edicion ? htmlspecialchars($anio_editar['anio']) : '' ?>" required>
                        </div>
                        <div class="mb-3">
                            <label class="form-label">Fecha Inicio <span class="text-danger">*</span></label>
                            <input type="date" name="fecha_inicio" class="form-control"
                                   value="<?= $modo_edicion ? htmlspecialchars($anio_editar['fecha_inicio']) : '' ?>" required>
                        </div>
                        <div class="mb-3">
                            <label class="form-label">Fecha Fin <span class="text-danger">*</span></label>
                            <input type="date" name="fecha_fin" class="form-control"
                                   value="<?= $modo_edicion ? htmlspecialchars($anio_editar['fecha_fin']) : '' ?>" required>
                        </div>
                        <?php if (!$modo_edicion): ?>
                        <div class="mb-3">
                            <label class="form-label">Estado</label>
                            <select name="estado" class="form-select">
                                <option value="inactivo" selected>Inactivo</option>
                                <option value="activo">Activo</option>
                            </select>
                            <small class="text-muted">Solo puede haber un año lectivo activo a la vez</small>
                        </div>
                        <?php endif; ?>
                        <?php if ($modo_edicion): ?>
                            <div class="d-flex gap-2">
                                <button type="submit" class="btn btn-primary flex-fill"><i class="bi bi-save me-2"></i>Guardar</button>
                                <a href="anios_lectivos.php" class="btn btn-secondary">Cancelar</a>
                            </div>
                        <?php else: ?>
                            <button type="submit" class="btn btn-primary w-100"><i class="bi bi-plus-circle me-2"></i>Crear</button>
                        <?php endif; ?>
                    </form>
                </div>
            </div>

            <div class="col-md-8">
                <div class="content-card">
                    <h5 class="mb-3">Años Lectivos Registrados</h5>
                    <div class="table-responsive">
                        <table class="table table-hover">
                            <thead>
                                <tr>
                                    <th>Año</th>
                                    <th>Fecha Inicio</th>
                                    <th>Fecha Fin</th>
                                    <th>Estado</th>
                                    <th>Acciones</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($anios as $a): ?>
                                <tr>
                                    <td><strong><?= htmlspecialchars($a['anio']) ?></strong></td>
                                    <td><?= formatearFecha($a['fecha_inicio']) ?></td>
                                    <td><?= formatearFecha($a['fecha_fin']) ?></td>
                                    <td>
                                        <?php if ($a['estado'] === 'activo'): ?>
                                            <span class="badge badge-active">Activo</span>
                                        <?php else: ?>
                                            <span class="badge bg-secondary">Inactivo</span>
                                        <?php endif; ?>
                                    </td>
                                    <td>
                                        <a href="anios_lectivos.php?editar=<?= $a['id'] ?>" class="btn btn-sm btn-primary" title="Editar"><i class="bi bi-pencil"></i></a>
                                        <?php if ($a['estado'] !== 'activo'): ?>
                                            <form method="POST" class="d-inline" onsubmit="return confirm('¿Activar este año lectivo?')">
                                                <input type="hidden" name="accion" value="activar">
                                                <input type="hidden" name="id" value="<?= $a['id'] ?>">
                                                <button class="btn btn-sm btn-success" title="Activar"><i class="bi bi-check-circle"></i></button>
                                            </form>
                                        <?php endif; ?>
                                        <form method="POST" class="d-inline" onsubmit="return confirm('¿Eliminar este año lectivo? Esta acción no se puede deshacer.')">
                                            <input type="hidden" name="accion" value="eliminar">
                                            <input type="hidden" name="id" value="<?= $a['id'] ?>">
                                            <button class="btn btn-sm btn-danger" title="Eliminar" <?= $a['estado'] === 'activo' ? 'disabled' : '' ?>><i class="bi bi-trash"></i></button>
                                        </form>
                                    </td>
                                </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>
