<?php
require_once '../config/db.php';
require_once '../config/functions.php';

requerirRol(['admin']);

require_once '../vendor/autoload.php';

use Shuchkin\SimpleXLSX;

$mensaje = '';
$tipo_mensaje = '';
$errores = [];
$registros_exitosos = 0;
$registros_fallidos = 0;

if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_FILES['archivo'])) {
    $archivo = $_FILES['archivo'];

    // Validar archivo
    if ($archivo['error'] !== UPLOAD_ERR_OK) {
        $mensaje = 'Error al subir el archivo';
        $tipo_mensaje = 'danger';
    } else {
        $extension = strtolower(pathinfo($archivo['name'], PATHINFO_EXTENSION));
        if ($extension !== 'xlsx') {
            $mensaje = 'Solo se permiten archivos .xlsx (Excel 2007+)';
            $tipo_mensaje = 'danger';
        } else {
            try {
                // Leer el archivo Excel
                $xlsx = SimpleXLSX::parse($archivo['tmp_name']);

                if (!$xlsx) {
                    throw new Exception(SimpleXLSX::parseError());
                }

                // Obtener encabezados (primera fila)
                $rows = $xlsx->rows();
                if (count($rows) < 2) {
                    throw new Exception('El archivo no tiene datos suficientes');
                }

                $headers = array_map('strtolower', array_map('trim', $rows[0]));

                // Mapeo de columnas
                $column_map = [
                    'dni' => array_search('dni', $headers) ?: array_search('dni *', $headers),
                    'apellido_paterno' => array_search('apellido paterno', $headers) ?: array_search('apellido paterno *', $headers),
                    'apellido_materno' => array_search('apellido materno', $headers) ?: array_search('apellido materno *', $headers),
                    'nombres' => array_search('nombres', $headers) ?: array_search('nombres *', $headers),
                    'email' => array_search('email', $headers) ?: array_search('email *', $headers),
                    'telefono' => array_search('telefono', $headers) ?: array_search('telefono *', $headers),
                    'direccion' => array_search('direccion', $headers),
                    'fecha_nacimiento' => array_search('fecha nacimiento', $headers),
                    'genero' => array_search('genero', $headers) ?: array_search('género', $headers),
                    'turno' => array_search('turno', $headers),
                ];

                // Validar columnas obligatorias
                $required_columns = ['dni', 'apellido_paterno', 'apellido_materno', 'nombres', 'email', 'telefono'];
                $missing_columns = [];
                foreach ($required_columns as $col) {
                    if ($column_map[$col] === false) {
                        $missing_columns[] = $col;
                    }
                }

                if (!empty($missing_columns)) {
                    throw new Exception('Faltan columnas obligatorias: ' . implode(', ', $missing_columns));
                }

                // Procesar filas (empezando desde la fila 2, saltando encabezados)
                $pdo->beginTransaction();
                $fila_num = 2; // Empezar en fila 2 (Excel)

                for ($i = 1; $i < count($rows); $i++) {
                    $row = $rows[$i];

                    $dni = trim($row[$column_map['dni']] ?? '');
                    $apellido_paterno = trim($row[$column_map['apellido_paterno']] ?? '');
                    $apellido_materno = trim($row[$column_map['apellido_materno']] ?? '');
                    $nombres = trim($row[$column_map['nombres']] ?? '');
                    $email = trim($row[$column_map['email']] ?? '');
                    $telefono = trim($row[$column_map['telefono']] ?? '');
                    $direccion = trim($row[$column_map['direccion']] ?? '');
                    $fecha_nacimiento = trim($row[$column_map['fecha_nacimiento']] ?? '');
                    $genero = trim($row[$column_map['genero']] ?? '');
                    $turno_nombre = trim(strtolower($row[$column_map['turno']] ?? ''));

                    // Convertir nombre de turno a id_turno
                    $id_turno = null;
                    if (!empty($turno_nombre)) {
                        $turno_info = query("SELECT id FROM turnos WHERE LOWER(nombre) = :turno AND estado = 'activo' LIMIT 1",
                            ['turno' => $turno_nombre])->fetch();
                        if ($turno_info) {
                            $id_turno = $turno_info['id'];
                        } else {
                            $error_fila[] = "Turno '$turno_nombre' no encontrado";
                        }
                    }

                    // Validaciones
                    $error_fila = [];

                    if (empty($dni)) {
                        $error_fila[] = 'DNI vacío';
                    } elseif (!preg_match('/^\d{8}$/', $dni)) {
                        $error_fila[] = 'DNI inválido (debe tener 8 dígitos)';
                    }

                    if (empty($apellido_paterno)) $error_fila[] = 'Apellido paterno vacío';
                    if (empty($apellido_materno)) $error_fila[] = 'Apellido materno vacío';
                    if (empty($nombres)) $error_fila[] = 'Nombres vacíos';
                    if (empty($email)) $error_fila[] = 'Email vacío';
                    if (empty($telefono)) $error_fila[] = 'Teléfono vacío';

                    if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
                        $error_fila[] = 'Email inválido';
                    }

                    // Convertir fecha de dd/mm/aaaa a aaaa-mm-dd
                    if (!empty($fecha_nacimiento)) {
                        $fecha_nacimiento = trim($fecha_nacimiento);
                        if (preg_match('/^\d{2}\/\d{2}\/\d{4}$/', $fecha_nacimiento)) {
                            $partes = explode('/', $fecha_nacimiento);
                            $fecha_nacimiento = $partes[2] . '-' . $partes[1] . '-' . $partes[0];
                        } elseif (!preg_match('/^\d{4}-\d{2}-\d{2}$/', $fecha_nacimiento)) {
                            $error_fila[] = 'Formato de fecha inválido (use dd/mm/aaaa)';
                        }
                    }

                    // Validar genero
                    if (!empty($genero) && !in_array(strtoupper($genero), ['M', 'F'])) {
                        $error_fila[] = 'Género inválido (debe ser M o F)';
                    } else {
                        $genero = strtoupper($genero);
                    }

                    if (!empty($error_fila)) {
                        $errores[] = "Fila $fila_num: " . implode(', ', $error_fila);
                        $registros_fallidos++;
                        $fila_num++;
                        continue;
                    }

                    // Verificar duplicados
                    $check_auxiliar = query("SELECT COUNT(*) as c FROM auxiliares WHERE dni = :dni", ['dni' => $dni])->fetch();
                    if ($check_auxiliar['c'] > 0) {
                        $errores[] = "Fila $fila_num: Ya existe un auxiliar con DNI $dni";
                        $registros_fallidos++;
                        $fila_num++;
                        continue;
                    }

                    $check_usuario = query("SELECT COUNT(*) as c FROM usuarios WHERE usuario = :usuario", ['usuario' => $dni])->fetch();
                    if ($check_usuario['c'] > 0) {
                        $errores[] = "Fila $fila_num: Ya existe un usuario con DNI $dni";
                        $registros_fallidos++;
                        $fila_num++;
                        continue;
                    }

                    // Crear usuario
                    $password = password_hash($dni, PASSWORD_DEFAULT);
                    $sql_usuario = "INSERT INTO usuarios (usuario, password, rol, estado) VALUES (:usuario, :password, 'auxiliar', 'activo')";
                    $stmt_usuario = $pdo->prepare($sql_usuario);
                    $stmt_usuario->execute([
                        'usuario' => $dni,
                        'password' => $password
                    ]);
                    $id_usuario = $pdo->lastInsertId();

                    // Crear auxiliar
                    $sql_auxiliar = "INSERT INTO auxiliares (id_usuario, dni, apellido_paterno, apellido_materno, nombres, email, telefono, direccion, fecha_nacimiento, genero, id_turno)
                                   VALUES (:id_usuario, :dni, :ap, :am, :nombres, :email, :telefono, :direccion, :fecha_nacimiento, :genero, :turno)";
                    $stmt_auxiliar = $pdo->prepare($sql_auxiliar);
                    $stmt_auxiliar->execute([
                        'id_usuario' => $id_usuario,
                        'dni' => $dni,
                        'ap' => $apellido_paterno,
                        'am' => $apellido_materno,
                        'nombres' => $nombres,
                        'email' => $email,
                        'telefono' => $telefono,
                        'direccion' => $direccion ?: null,
                        'fecha_nacimiento' => $fecha_nacimiento ?: null,
                        'genero' => $genero ?: null,
                        'turno' => $id_turno
                    ]);

                    $registros_exitosos++;
                    $fila_num++;
                }

                $pdo->commit();

                if (empty($errores)) {
                    $mensaje = "Importación completada exitosamente. $registros_exitosos auxiliares importados.";
                    $tipo_mensaje = 'success';
                } else {
                    $mensaje = "Importación finalizada con advertencias. Exitosos: $registros_exitosos, Fallidos: $registros_fallidos";
                    $tipo_mensaje = 'warning';
                }

            } catch (Exception $e) {
                if ($pdo->inTransaction()) {
                    $pdo->rollBack();
                }
                $mensaje = 'Error al procesar el archivo: ' . $e->getMessage();
                $tipo_mensaje = 'danger';
            }
        }
    }
} else {
    header('Location: importar.php');
    exit;
}
?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0, maximum-scale=1.0, user-scalable=no">
    <title>SICA - Resultado de Importación</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.1/font/bootstrap-icons.css" rel="stylesheet">
    <link href="../assets/css/sica.css?v=5.1" rel="stylesheet">
</head>
<body>
    <nav class="sidebar">
        <?php include '../includes/sidebar.php'; ?>
    </nav>

    <div class="main-content">
        <div class="d-flex justify-content-between align-items-center mb-4">
            <div>
                <h3 class="mb-1">Resultado de Importación</h3>
                <p class="text-muted mb-0">Resumen del proceso de importación</p>
            </div>
            <a href="index.php" class="btn btn-outline-secondary">
                <i class="bi bi-arrow-left"></i> Volver a Auxiliares
            </a>
        </div>

        <div class="alert alert-<?= $tipo_mensaje ?> alert-dismissible fade show" role="alert">
            <?= htmlspecialchars($mensaje) ?>
            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
        </div>

        <!-- Estadísticas -->
        <div class="row mb-4">
            <div class="col-md-6">
                <div class="card stats-card border-0">
                    <div class="card-body">
                        <div class="d-flex align-items-center">
                            <div class="stats-icon bg-success">
                                <i class="bi bi-check-circle-fill"></i>
                            </div>
                            <div class="ms-3">
                                <h6 class="mb-0 text-muted">Exitosos</h6>
                                <h3 class="mb-0"><?= $registros_exitosos ?></h3>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
            <div class="col-md-6">
                <div class="card stats-card border-0">
                    <div class="card-body">
                        <div class="d-flex align-items-center">
                            <div class="stats-icon bg-danger">
                                <i class="bi bi-x-circle-fill"></i>
                            </div>
                            <div class="ms-3">
                                <h6 class="mb-0 text-muted">Fallidos</h6>
                                <h3 class="mb-0"><?= $registros_fallidos ?></h3>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <?php if (!empty($errores)): ?>
            <div class="card">
                <div class="card-header bg-danger text-white">
                    <h5 class="mb-0"><i class="bi bi-exclamation-triangle"></i> Errores Encontrados (<?= count($errores) ?>)</h5>
                </div>
                <div class="card-body">
                    <div class="table-responsive">
                        <table class="table table-striped">
                            <thead>
                                <tr>
                                    <th>#</th>
                                    <th>Error</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($errores as $i => $error): ?>
                                    <tr>
                                        <td><?= $i + 1 ?></td>
                                        <td><?= htmlspecialchars($error) ?></td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
        <?php endif; ?>

        <div class="d-flex gap-2 mt-4">
            <a href="importar.php" class="btn btn-primary">
                <i class="bi bi-upload"></i> Importar Más
            </a>
            <a href="index.php" class="btn btn-outline-secondary">
                <i class="bi bi-list"></i> Ver Auxiliares
            </a>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>
