<?php
require_once '../config/db.php';
require_once '../config/functions.php';

requerirRol(['admin']);

$id = intval($_GET['id'] ?? 0);

if (!$id) {
    header('Location: index.php');
    exit;
}

// Obtener auxiliar
$auxiliar = query("SELECT a.*, u.usuario FROM auxiliares a LEFT JOIN usuarios u ON a.id_usuario = u.id WHERE a.id = :id", ['id' => $id])->fetch();

if (!$auxiliar) {
    header('Location: index.php');
    exit;
}

$mensaje = '';
$tipo_mensaje = '';
$eliminado = false;

// Obtener año lectivo seleccionado
$anio_lectivo_id = getAnioLectivoSeleccionado();

// Verificar asignaciones del auxiliar para mostrar advertencias
$programaciones_asignadas = query("
    SELECT COUNT(*) as total
    FROM programaciones
    WHERE id_auxiliar = ?
    AND id_anio_lectivo = ?
    AND estado = 'activo'
", [$id, $anio_lectivo_id])->fetch();

$tiene_asignaciones = ($programaciones_asignadas['total'] > 0);

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $accion = $_POST['accion'] ?? '';

    if ($accion === 'eliminar') {
        try {
            // VALIDACIÓN: Verificar si es auxiliar en alguna programación del año lectivo actual
            if ($tiene_asignaciones) {
                $mensaje = "No se puede eliminar al auxiliar porque está asignado a {$programaciones_asignadas['total']} programación(es) en el año lectivo actual. Primero debe reasignar los auxiliares.";
                $tipo_mensaje = 'danger';
            } else {
                // Iniciar transacción
                $pdo->beginTransaction();

                // Eliminar usuario (esto CASCADE eliminará el auxiliar)
                query("DELETE FROM usuarios WHERE id = :id", ['id' => $auxiliar['id_usuario']]);

                $pdo->commit();
                $mensaje = 'Auxiliar eliminado exitosamente';
                $tipo_mensaje = 'success';
                $eliminado = true;
            }
        } catch (Exception $e) {
            $pdo->rollBack();
            $mensaje = 'Error al eliminar: ' . $e->getMessage();
            $tipo_mensaje = 'danger';
        }
    } elseif ($accion === 'cambiar_estado') {
        $nuevo_estado = $_POST['nuevo_estado'] ?? 'inactivo';
        try {
            // Actualizar estado del auxiliar
            query("UPDATE auxiliares SET estado = :estado WHERE id = :id",
                  ['estado' => $nuevo_estado, 'id' => $id]);

            // Actualizar estado del usuario
            $usuario_estado = ($nuevo_estado === 'activo') ? 'activo' : 'inactivo';
            query("UPDATE usuarios SET estado = :estado WHERE id = :id",
                  ['estado' => $usuario_estado, 'id' => $auxiliar['id_usuario']]);

            $mensaje = 'Estado actualizado exitosamente';
            $tipo_mensaje = 'success';

            // Recargar datos
            $auxiliar = query("SELECT a.*, u.usuario FROM auxiliares a LEFT JOIN usuarios u ON a.id_usuario = u.id WHERE a.id = :id", ['id' => $id])->fetch();
        } catch (Exception $e) {
            $mensaje = 'Error al cambiar estado: ' . $e->getMessage();
            $tipo_mensaje = 'danger';
        }
    }
}
?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0, maximum-scale=1.0, user-scalable=no">
    <title>SICA - Eliminar Auxiliar</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.1/font/bootstrap-icons.css" rel="stylesheet">
    <link href="../assets/css/sica.css?v=5.1" rel="stylesheet">
</head>
<body>
    <nav class="sidebar">
        <?php include '../includes/sidebar.php'; ?>
    </nav>

    <div class="main-content">
        <div class="d-flex justify-content-between align-items-center mb-4">
            <div>
                <h3 class="mb-1">Eliminar Auxiliar</h3>
                <p class="text-muted mb-0">Gestión de eliminación de auxiliares</p>
            </div>
            <a href="index.php" class="btn btn-outline-secondary">
                <i class="bi bi-arrow-left"></i> Volver
            </a>
        </div>

        <?php if ($mensaje): ?>
            <div class="alert alert-<?= $tipo_mensaje ?> alert-dismissible fade show" role="alert">
                <?= htmlspecialchars($mensaje) ?>
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            </div>
        <?php endif; ?>

        <?php if (!$eliminado): ?>
            <div class="row">
                <div class="col-lg-6">
                    <div class="card">
                        <div class="card-header bg-danger text-white">
                            <h5 class="mb-0"><i class="bi bi-exclamation-triangle"></i> Confirmar Eliminación</h5>
                        </div>
                        <div class="card-body">
                            <?php if ($tiene_asignaciones): ?>
                                <div class="alert alert-danger">
                                    <i class="bi bi-x-circle"></i>
                                    <strong>No se puede eliminar este auxiliar</strong>
                                    <hr class="my-2">
                                    <p class="mb-1">El auxiliar tiene las siguientes asignaciones en el año lectivo actual:</p>
                                    <ul class="mb-0">
                                        <li><strong>Auxiliar de <?= $programaciones_asignadas['total'] ?> programación(es)</strong></li>
                                    </ul>
                                    <p class="mb-0 mt-2"><small>Debe reasignar estas relaciones antes de eliminar.</small></p>
                                </div>
                            <?php else: ?>
                                <div class="alert alert-warning">
                                    <i class="bi bi-exclamation-circle"></i>
                                    <strong>Advertencia:</strong> Esta acción eliminará permanentemente al auxiliar y su usuario del sistema. Esta acción no se puede deshacer.
                                </div>
                            <?php endif; ?>

                            <div class="mb-4">
                                <h6>Auxiliar a eliminar:</h6>
                                <table class="table table-bordered">
                                    <tr>
                                        <td width="30%"><strong>DNI:</strong></td>
                                        <td><?= htmlspecialchars($auxiliar['dni']) ?></td>
                                    </tr>
                                    <tr>
                                        <td><strong>Nombre:</strong></td>
                                        <td>
                                            <?= htmlspecialchars($auxiliar['apellido_paterno'] . ' ' . $auxiliar['apellido_materno'] . ', ' . $auxiliar['nombres']) ?>
                                        </td>
                                    </tr>
                                    <tr>
                                        <td><strong>Usuario:</strong></td>
                                        <td><?= htmlspecialchars($auxiliar['usuario']) ?></td>
                                    </tr>
                                    <tr>
                                        <td><strong>Estado:</strong></td>
                                        <td>
                                            <span class="badge <?= $auxiliar['estado'] === 'activo' ? 'bg-success' : 'bg-secondary' ?>">
                                                <?= ucfirst($auxiliar['estado']) ?>
                                            </span>
                                        </td>
                                    </tr>
                                </table>
                            </div>

                            <form method="POST">
                                <input type="hidden" name="accion" value="eliminar">
                                <div class="d-flex gap-2">
                                    <button type="submit" class="btn btn-danger" <?= $tiene_asignaciones ? 'disabled' : '' ?>>
                                        <i class="bi bi-trash"></i> Sí, Eliminar Auxiliar
                                    </button>
                                    <a href="index.php" class="btn btn-secondary">
                                        <i class="bi bi-x-circle"></i> Cancelar
                                    </a>
                                </div>
                                <?php if ($tiene_asignaciones): ?>
                                    <div class="form-text text-muted mt-2">
                                        <i class="bi bi-info-circle"></i> El botón está deshabilitado porque el auxiliar tiene asignaciones activas.
                                    </div>
                                <?php endif; ?>
                            </form>
                        </div>
                    </div>
                </div>

                <div class="col-lg-6">
                    <div class="card mb-3">
                        <div class="card-header bg-warning">
                            <h5 class="mb-0"><i class="bi bi-lightbulb"></i> Alternativas</h5>
                        </div>
                        <div class="card-body">
                            <p>En lugar de eliminar al auxiliar, considere estas opciones:</p>

                            <h6>Cambiar a Inactivo</h6>
                            <p class="small text-muted">Mantiene los datos pero impide el acceso al sistema.</p>
                            <form method="POST" class="mb-3">
                                <input type="hidden" name="accion" value="cambiar_estado">
                                <input type="hidden" name="nuevo_estado" value="inactivo">
                                <button type="submit" class="btn btn-outline-warning">
                                    <i class="bi bi-dash-circle"></i> Cambiar a Inactivo
                                </button>
                            </form>

                            <h6>Cambiar a Retirado</h6>
                            <p class="small text-muted">Marca al auxiliar como retirado de la institución.</p>
                            <form method="POST">
                                <input type="hidden" name="accion" value="cambiar_estado">
                                <input type="hidden" name="nuevo_estado" value="retirado">
                                <button type="submit" class="btn btn-outline-secondary">
                                    <i class="bi bi-person-x"></i> Cambiar a Retirado
                                </button>
                            </form>
                        </div>
                    </div>
                </div>
            </div>
        <?php else: ?>
            <div class="text-center py-5">
                <i class="bi bi-check-circle text-success" style="font-size: 4rem;"></i>
                <h4 class="mt-3">Auxiliar Eliminado</h4>
                <p class="text-muted">El auxiliar ha sido eliminado exitosamente del sistema.</p>
                <a href="index.php" class="btn btn-primary">
                    <i class="bi bi-arrow-left"></i> Volver al Listado
                </a>
            </div>
        <?php endif; ?>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>
