/**
 * Script de control del Sidebar para SICA
 * Maneja el comportamiento responsive y persistencia del estado
 */

(function() {
    'use strict';

    // Recuperar estado del sidebar desde localStorage al cargar
    document.addEventListener('DOMContentLoaded', function() {
        const sidebar = document.getElementById('sidebar');
        const mainContent = document.getElementById('mainContent');
        const toggleBtn = document.getElementById('sidebarToggleBtn');
        const overlay = document.getElementById('sidebarOverlay');

        if (!sidebar) return; // Salir si no hay sidebar en la página

        // PC: Sidebar expandido por defecto en PC
        if (window.innerWidth >= 992) {
            sidebar.classList.remove('collapsed');
            if (mainContent) mainContent.classList.remove('expanded');
            if (toggleBtn) {
                toggleBtn.classList.remove('collapsed');
                const icon = toggleBtn.querySelector('i');
                if (icon) icon.className = 'bi bi-chevron-left';
            }
            // Limpiar localStorage para asegurar estado expandido
            localStorage.setItem('sidebarCollapsed', 'false');
        }

        // Móvil: Recuperar estado visible/oculto
        const isMobileVisible = localStorage.getItem('sidebarMobileVisible') === 'true';
        if (isMobileVisible && window.innerWidth < 992) {
            sidebar.classList.add('show');
            if (overlay) overlay.classList.add('active');
        }

        // Agregar event listeners a los links del sidebar para móvil
        const sidebarLinks = sidebar.querySelectorAll('a[href]');
        sidebarLinks.forEach(function(link) {
            link.addEventListener('click', function() {
                if (window.innerWidth < 992) {
                    // Cerrar sidebar en móvil al hacer clic en un link
                    closeSidebarMobile();
                }
            });
        });
    });

    // Toggle para PC (colapsar/expandir)
    window.toggleSidebarPC = function() {
        const sidebar = document.getElementById('sidebar');
        const mainContent = document.getElementById('mainContent');
        const toggleBtn = document.getElementById('sidebarToggleBtn');

        if (!sidebar) return;

        sidebar.classList.toggle('collapsed');
        if (mainContent) mainContent.classList.toggle('expanded');
        if (toggleBtn) toggleBtn.classList.toggle('collapsed');

        const isCollapsed = sidebar.classList.contains('collapsed');
        localStorage.setItem('sidebarCollapsed', isCollapsed);

        if (toggleBtn) {
            const icon = toggleBtn.querySelector('i');
            if (icon) {
                icon.className = isCollapsed ? 'bi bi-chevron-right' : 'bi bi-chevron-left';
            }
        }
    };

    // Toggle para móvil (mostrar/ocultar)
    window.toggleSidebarMobile = function() {
        const sidebar = document.getElementById('sidebar');
        const overlay = document.getElementById('sidebarOverlay');

        if (!sidebar) return;

        sidebar.classList.toggle('show');
        if (overlay) overlay.classList.toggle('active');

        const isVisible = sidebar.classList.contains('show');
        localStorage.setItem('sidebarMobileVisible', isVisible);
    };

    // Cerrar sidebar en móvil
    window.closeSidebarMobile = function() {
        const sidebar = document.getElementById('sidebar');
        const overlay = document.getElementById('sidebarOverlay');

        if (!sidebar) return;

        sidebar.classList.remove('show');
        if (overlay) overlay.classList.remove('active');
        localStorage.setItem('sidebarMobileVisible', 'false');
    };

    // Manejar cambio de tamaño de ventana
    window.addEventListener('resize', function() {
        const sidebar = document.getElementById('sidebar');
        const overlay = document.getElementById('sidebarOverlay');

        if (!sidebar) return;

        // Si pasamos de móvil a PC, limpiar estados móviles
        if (window.innerWidth >= 992) {
            sidebar.classList.remove('show');
            if (overlay) overlay.classList.remove('active');
        }
    });
})();
