<?php
require_once '../config/db.php';
require_once '../config/functions.php';

requerirRol(['admin', 'auxiliar', 'docente']);

$anio_seleccionado_id = getAnioLectivoSeleccionado();

if (!$anio_seleccionado_id) {
    die('Error: No hay año lectivo seleccionado. Configure un año lectivo activo primero.');
}

// Obtener estados de asistencia
$estados_asistencia = query("SELECT * FROM estados_asistencia WHERE estado = 'activo' ORDER BY nombre")->fetchAll();

// Obtener filtros seleccionados
$id_nivel = $_GET['nivel'] ?? '';
$id_grado = $_GET['grado'] ?? '';
$id_seccion = $_GET['seccion'] ?? '';
$fecha_registro = $_GET['fecha'] ?? date('Y-m-d');

// Construir consulta para estudiantes (similar al reporte)
$sql_estudiantes = "
    SELECT DISTINCT
        e.id,
        e.dni,
        e.apellido_paterno,
        e.apellido_materno,
        e.nombres,
        e.genero,
        pe.id_programacion,
        p.id_seccion,
        p.id_turno,
        s.seccion,
        g.grado,
        n.id as nivel_id,
        n.nombre as nivel_nombre,
        t.nombre as turno_nombre,
        t.abreviatura as turno_abreviatura,
        t.hora_ingreso,
        t.hora_salida,
        t.tolerancia_minutos,
        ea.id as id_asistencia,
        ea.id_estado_asistencia,
        ea.estado_codigo,
        ea.hora_registro as asistencia_hora,
        ea.minutos_tardanza
    FROM estudiantes e
    INNER JOIN programaciones_estudiantes pe ON e.id = pe.id_estudiante
        AND pe.estado = 'activo'
    INNER JOIN programaciones p ON pe.id_programacion = p.id
        AND p.id_anio_lectivo = ? AND p.estado = 'activo'
    INNER JOIN secciones s ON p.id_seccion = s.id
    INNER JOIN grados g ON s.id_grado = g.id
    INNER JOIN niveles n ON g.id_nivel = n.id
    INNER JOIN turnos t ON p.id_turno = t.id
    LEFT JOIN asistencias ea ON e.id = ea.id_estudiante
        AND ea.fecha = ?
        AND ea.id_anio_lectivo = ?
    WHERE e.estado = 'activo'
";

$params = [$anio_seleccionado_id, $fecha_registro, $anio_seleccionado_id];

// Aplicar filtros
if ($id_nivel) {
    $sql_estudiantes .= " AND n.id = ?";
    $params[] = $id_nivel;
}
if ($id_grado) {
    $sql_estudiantes .= " AND g.id = ?";
    $params[] = $id_grado;
}
if ($id_seccion) {
    $sql_estudiantes .= " AND s.id = ?";
    $params[] = $id_seccion;
}

$sql_estudiantes .= " ORDER BY n.id, g.id, s.seccion, e.apellido_paterno, e.apellido_materno, e.nombres";

$estudiantes = query($sql_estudiantes, $params)->fetchAll();

// Obtener lista de niveles para filtros
$niveles = query("SELECT DISTINCT n.id, n.nombre FROM niveles n
    INNER JOIN grados g ON n.id = g.id_nivel
    INNER JOIN secciones s ON g.id = s.id_grado
    INNER JOIN programaciones p ON s.id = p.id_seccion
    WHERE p.id_anio_lectivo = ? AND p.estado = 'activo'
    ORDER BY n.id", [$anio_seleccionado_id])->fetchAll();

// Obtener grados según nivel seleccionado
$grados = [];
if ($id_nivel) {
    $grados = query("SELECT DISTINCT g.id, g.grado FROM grados g
        INNER JOIN secciones s ON g.id = s.id_grado
        INNER JOIN programaciones p ON s.id = p.id_seccion
        WHERE p.id_anio_lectivo = ? AND g.id_nivel = ? AND p.estado = 'activo'
        ORDER BY g.id", [$anio_seleccionado_id, $id_nivel])->fetchAll();
}

// Obtener secciones según grado seleccionado
$secciones = [];
if ($id_grado) {
    $secciones = query("SELECT DISTINCT s.id, s.seccion FROM secciones s
        INNER JOIN programaciones p ON s.id = p.id_seccion
        WHERE p.id_anio_lectivo = ? AND s.id_grado = ? AND p.estado = 'activo'
        ORDER BY s.seccion", [$anio_seleccionado_id, $id_grado])->fetchAll();
}

// Calcular estadísticas
$total_estudiantes = count($estudiantes);
$con_asistencia = 0;
$sin_asistencia = 0;

foreach ($estudiantes as $est) {
    if ($est['id_asistencia']) {
        $con_asistencia++;
    } else {
        $sin_asistencia++;
    }
}

$porcentaje_registrado = $total_estudiantes > 0 ? round(($con_asistencia / $total_estudiantes) * 100, 1) : 0;
?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0, maximum-scale=1.0, user-scalable=no">
    <title>SICA - Registro Grupal de Asistencias</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.1/font/bootstrap-icons.css" rel="stylesheet">
    <link href="../assets/css/sica.css?v=5.1" rel="stylesheet">
    <style>
        .asistencia-table {
            background: white;
            border-radius: 10px;
            overflow: hidden;
            box-shadow: 0 2px 8px rgba(0,0,0,0.08);
            margin-bottom: 0;
        }
        .asistencia-table thead {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
        }
        .asistencia-table th {
            font-weight: 600;
            text-align: center;
            vertical-align: middle;
            padding: 10px 8px;
            border: none;
            white-space: nowrap;
            font-size: 0.85rem;
        }
        .asistencia-table td {
            padding: 8px;
            vertical-align: middle;
            border-bottom: 1px solid #f3f4f6;
            font-size: 0.9rem;
        }
        .asistencia-table tbody tr:hover {
            background: #fafafa;
        }
        .asistencia-table tbody tr:nth-child(even) {
            background: #f9fafb;
        }
        .asistencia-table tbody tr:nth-child(even):hover {
            background: #f0f0f0;
        }
        .celda-nro {
            width: 4%;
            text-align: center;
            font-weight: bold;
            color: #64748b;
        }
        .celda-nombre {
            font-weight: 500;
            color: #1e293b;
        }
        .celda-dni {
            color: #64748b;
            font-size: 0.85rem;
        }
        .estado-badge {
            display: inline-block;
            padding: 3px 8px;
            border-radius: 10px;
            font-size: 0.7rem;
            font-weight: 600;
            text-align: center;
            min-width: 35px;
        }
        .badge-registrado {
            background: #10b981;
            color: white;
        }
        .badge-pendiente {
            background: #f59e0b;
            color: white;
        }
        .section-header {
            background: linear-gradient(135deg, #3b82f6 0%, #8b5cf6 100%);
            color: white;
            padding: 12px 15px;
            border-radius: 8px;
            margin-bottom: 15px;
            font-weight: 600;
            font-size: 1rem;
            display: flex;
            justify-content: space-between;
            align-items: center;
        }
        .section-info {
            font-size: 0.85rem;
            opacity: 0.9;
        }
        .stats-card {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            padding: 15px;
            border-radius: 10px;
            margin-bottom: 20px;
        }
        .stats-card .h3 {
            font-size: 1.8rem;
            margin-bottom: 0;
        }
        .stats-card small {
            font-size: 0.75rem;
            opacity: 0.9;
        }
        .hora-input {
            width: 95px;
            text-align: center;
            font-weight: 500;
            font-size: 0.9rem;
            padding: 4px 8px;
        }
        .estado-select {
            min-width: 150px;
            font-size: 0.85rem;
            padding: 4px 8px;
        }
        .loading-overlay {
            display: none;
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background: rgba(0,0,0,0.5);
            z-index: 9999;
            justify-content: center;
            align-items: center;
        }
        .loading-overlay.show {
            display: flex;
        }
        .loading-spinner {
            background: white;
            padding: 30px;
            border-radius: 15px;
            text-align: center;
        }
        .btn-save-fixed {
            position: fixed;
            bottom: 25px;
            right: 25px;
            z-index: 100;
            box-shadow: 0 4px 20px rgba(0,0,0,0.3);
        }
        .grupo-seccion {
            margin-bottom: 25px;
        }
        .table-responsive {
            border-radius: 10px;
            overflow: hidden;
        }
        .table-responsive::-webkit-scrollbar {
            height: 8px;
        }
        .table-responsive::-webkit-scrollbar-track {
            background: #f1f1f1;
            border-radius: 0 0 10px 10px;
        }
        .table-responsive::-webkit-scrollbar-thumb {
            background: #888;
            border-radius: 4px;
        }
        .table-responsive::-webkit-scrollbar-thumb:hover {
            background: #555;
        }
    </style>
</head>
<body>
    <?php include '../includes/layout_elems.php'; ?>

    <div class="d-flex justify-content-between align-items-center mb-4">
        <div class="d-flex align-items-center">
            <button class="btn btn-link d-lg-none me-3" onclick="toggleSidebarMobile()">
                <i class="bi bi-list fs-4"></i>
            </button>
            <div>
                <h3 class="mb-1"><i class="bi bi-people-fill"></i> Registro Grupal de Asistencias</h3>
                <p class="text-muted mb-0">Registre la asistencia de todos los estudiantes de una sección</p>
            </div>
        </div>
        <div>
            <a href="escanear_qr.php" class="btn btn-outline-primary me-2">
                <i class="bi bi-qr-code-scan"></i> Escáner QR
            </a>
            <a href="registrar_manual.php" class="btn btn-outline-secondary">
                <i class="bi bi-person-plus"></i> Registro Individual
            </a>
        </div>
    </div>

    <!-- Filtros -->
    <div class="card mb-4">
        <div class="card-body">
            <form method="GET" action="" class="row g-3">
                <div class="col-md-2">
                    <label class="form-label fw-bold">Fecha de Registro</label>
                    <input type="date" name="fecha" class="form-control" value="<?= htmlspecialchars($fecha_registro) ?>" required>
                </div>
                <div class="col-md-2">
                    <label class="form-label fw-bold">Nivel</label>
                    <select name="nivel" class="form-select" onchange="this.form.submit()">
                        <option value="">Todos</option>
                        <?php foreach ($niveles as $nivel): ?>
                            <option value="<?= $nivel['id'] ?>" <?= $id_nivel == $nivel['id'] ? 'selected' : '' ?>>
                                <?= htmlspecialchars($nivel['nombre']) ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>
                <div class="col-md-2">
                    <label class="form-label fw-bold">Grado</label>
                    <select name="grado" class="form-select" onchange="this.form.submit()" <?= !$id_nivel ? 'disabled' : '' ?>>
                        <option value="">Todos</option>
                        <?php foreach ($grados as $grado): ?>
                            <option value="<?= $grado['id'] ?>" <?= $id_grado == $grado['id'] ? 'selected' : '' ?>>
                                <?= htmlspecialchars($grado['grado']) ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>
                <div class="col-md-2">
                    <label class="form-label fw-bold">Sección</label>
                    <select name="seccion" class="form-select" onchange="this.form.submit()" <?= !$id_grado ? 'disabled' : '' ?>>
                        <option value="">Todas</option>
                        <?php foreach ($secciones as $seccion): ?>
                            <option value="<?= $seccion['id'] ?>" <?= $id_seccion == $seccion['id'] ? 'selected' : '' ?>>
                                <?= htmlspecialchars($seccion['seccion']) ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>
                <div class="col-md-2 d-flex align-items-end">
                    <button type="submit" class="btn btn-primary w-100">
                        <i class="bi bi-funnel"></i> Filtrar
                    </button>
                </div>
                <div class="col-md-2 d-flex align-items-end">
                    <a href="<?= $_SERVER['PHP_SELF'] ?>" class="btn btn-outline-secondary w-100">
                        <i class="bi bi-arrow-counterclockwise"></i> Limpiar
                    </a>
                </div>
            </form>
        </div>
    </div>

    <?php if ($estudiantes): ?>
        <!-- Estadísticas -->
        <div class="stats-card">
            <div class="row text-center">
                <div class="col-md-3">
                    <div class="h3 mb-1"><?= $total_estudiantes ?></div>
                    <small>Total Estudiantes</small>
                </div>
                <div class="col-md-3">
                    <div class="h3 mb-1"><?= $con_asistencia ?></div>
                    <small>Con Asistencia</small>
                </div>
                <div class="col-md-3">
                    <div class="h3 mb-1"><?= $sin_asistencia ?></div>
                    <small>Pendientes</small>
                </div>
                <div class="col-md-3">
                    <div class="h3 mb-1"><?= $porcentaje_registrado ?>%</div>
                    <small>Registrado</small>
                </div>
            </div>
        </div>

        <!-- Alertas -->
        <?php if ($sin_asistencia > 0): ?>
            <div class="alert alert-warning alert-dismissible fade show" role="alert">
                <i class="bi bi-exclamation-triangle-fill"></i>
                Hay <strong><?= $sin_asistencia ?></strong> estudiante(s) pendiente(s) de registrar.
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            </div>
        <?php endif; ?>

        <?php if ($con_asistencia == $total_estudiantes): ?>
            <div class="alert alert-success alert-dismissible fade show" role="alert">
                <i class="bi bi-check-circle-fill"></i>
                ¡Todos los estudiantes tienen su asistencia registrada!
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            </div>
        <?php endif; ?>

        <!-- Formulario de registro -->
        <form id="formAsistencias" action="ajax.php" method="POST">
            <input type="hidden" name="action" value="guardar_asistencias">
            <input type="hidden" name="fecha" value="<?= htmlspecialchars($fecha_registro) ?>">

            <?php
            $grupo_actual = '';
            $contador_grupo = 0;
            $estudiantes_por_grupo = [];

            // Agrupar estudiantes para contar
            foreach ($estudiantes as $est) {
                $grupo_key = $est['nivel_nombre'] . ' - ' . $est['grado'] . '° "' . $est['seccion'] . '" - Turno ' . $est['turno_abreviatura'];
                if (!isset($estudiantes_por_grupo[$grupo_key])) {
                    $estudiantes_por_grupo[$grupo_key] = [];
                }
                $estudiantes_por_grupo[$grupo_key][] = $est;
            }

            $indice_global = 0;
            foreach ($estudiantes as $est):
                $grupo_key = $est['nivel_nombre'] . ' - ' . $est['grado'] . '° "' . $est['seccion'] . '" - Turno ' . $est['turno_abreviatura'];

                if ($grupo_key != $grupo_actual):
                    if ($grupo_actual != ''): ?>
                        </table>
                        </div>
                    <?php endif; ?>

                    <?php
                    $total_grupo = count($estudiantes_por_grupo[$grupo_key]);
                    $registrados_grupo = 0;
                    foreach ($estudiantes_por_grupo[$grupo_key] as $e) {
                        if ($e['id_asistencia']) $registrados_grupo++;
                    }
                    ?>

                    <div class="grupo-seccion">
                        <div class="section-header">
                            <div>
                                <i class="bi bi-people"></i> <?= htmlspecialchars($grupo_key) ?>
                            </div>
                            <div class="section-info">
                                <?= $registrados_grupo ?>/<?= $total_grupo ?> registrados
                                <?php if ($total_grupo > 0): ?>
                                    (<?= round(($registrados_grupo / $total_grupo) * 100) ?>%)
                                <?php endif; ?>
                            </div>
                        </div>
                        <div class="table-responsive">
                        <table class="table table-striped asistencia-table mb-0">
                            <thead>
                                <tr>
                                    <th style="width: 5%;">#</th>
                                    <th style="width: 38%;">Estudiante</th>
                                    <th style="width: 13%;">DNI</th>
                                    <th style="width: 14%;">Hora</th>
                                    <th style="width: 25%;">Estado</th>
                                    <th style="width: 5%;">✓</th>
                                </tr>
                            </thead>
                            <tbody>
                    <?php
                            $contador_grupo = 0;
                            $grupo_actual = $grupo_key;
                endif;

                $contador_grupo++;
            ?>

                <tr data-id="<?= $est['id'] ?>">
                    <td class="celda-nro"><?= $contador_grupo ?></td>
                    <td class="celda-nombre">
                        <?= htmlspecialchars($est['apellido_paterno'] . ' ' . $est['apellido_materno'] . ', ' . $est['nombres']) ?>
                    </td>
                    <td class="celda-dni">
                        <?= htmlspecialchars($est['dni']) ?>
                    </td>
                    <td>
                        <input type="time"
                               name="asistencias[<?= $est['id'] ?>][hora]"
                               class="form-control hora-input"
                               value="<?= $est['asistencia_hora'] ?? date('H:i') ?>"
                               required>
                    </td>
                    <td>
                        <select name="asistencias[<?= $est['id'] ?>][estado]"
                                class="form-select estado-select"
                                onchange="actualizarBadge(this, <?= $est['id'] ?>)"
                                required>
                            <option value="">Seleccione...</option>
                            <?php foreach ($estados_asistencia as $estado): ?>
                                <option value="<?= $estado['id'] ?>"
                                        data-codigo="<?= htmlspecialchars($estado['abreviatura']) ?>"
                                        data-nombre="<?= htmlspecialchars($estado['nombre']) ?>"
                                        <?= $est['estado_codigo'] == $estado['abreviatura'] ? 'selected' : '' ?>>
                                    <?= htmlspecialchars($estado['nombre']) ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                        <!-- Campos ocultos dentro del TD -->
                        <input type="hidden" name="asistencias[<?= $est['id'] ?>][id_programacion]" value="<?= $est['id_programacion'] ?>">
                        <input type="hidden" name="asistencias[<?= $est['id'] ?>][id_turno]" value="<?= $est['id_turno'] ?>">
                    </td>
                    <td class="text-center">
                        <span id="badge-<?= $est['id'] ?>" class="estado-badge <?= $est['id_asistencia'] ? 'badge-registrado' : 'badge-pendiente' ?>">
                            <?= $est['id_asistencia'] ? '✓' : '—' ?>
                        </span>
                    </td>
                </tr>

            <?php endforeach; ?>

            <?php if ($grupo_actual != ''): ?>
                </tbody>
            </table>
            </div>
            </div>
            <?php endif; ?>

            <!-- Botón de guardar flotante -->
            <button type="submit" class="btn btn-primary btn-lg btn-save-fixed">
                <i class="bi bi-save"></i> Guardar Todas las Asistencias
                <span class="badge bg-white text-primary ms-2" id="contadorCambios">0</span>
            </button>
        </form>
    <?php else: ?>
        <div class="alert alert-info text-center">
            <i class="bi bi-info-circle"></i>
            <strong>No se encontraron estudiantes</strong>
            <br>
            Seleccione una sección específica para ver los estudiantes.
        </div>
    <?php endif; ?>

    <!-- Loading Overlay -->
    <div class="loading-overlay" id="loadingOverlay">
        <div class="loading-spinner">
            <div class="spinner-border text-primary mb-3" style="width: 3rem; height: 3rem;" role="status">
                <span class="visually-hidden">Cargando...</span>
            </div>
            <h5>Guardando asistencias...</h5>
            <p class="text-muted">Por favor espere</p>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        let cambiosPendientes = 0;

        document.getElementById('formAsistencias').addEventListener('submit', function(e) {
            e.preventDefault();

            // Confirmar si hay estudiantes sin estado seleccionado
            const selects = document.querySelectorAll('.estado-select');
            let sinEstado = 0;
            selects.forEach(select => {
                if (select.value === '') {
                    sinEstado++;
                }
            });

            if (sinEstado > 0) {
                if (!confirm(`Hay ${sinEstado} estudiante(s) sin estado de asistencia seleccionado. ¿Desea continuar?`)) {
                    return;
                }
            }

            // Mostrar loading
            document.getElementById('loadingOverlay').classList.add('show');

            const formData = new FormData(this);

            fetch('ajax.php', {
                method: 'POST',
                body: formData
            })
            .then(response => response.json())
            .then(data => {
                document.getElementById('loadingOverlay').classList.remove('show');

                if (data.success) {
                    // Mostrar alertas de éxito
                    if (data.nuevas > 0) {
                        showAlert('success', `✓ Se registraron ${data.nuevas} nuevas asistencias.`);
                    }
                    if (data.actualizadas > 0) {
                        showAlert('info', `✓ Se actualizaron ${data.actualizadas} asistencias existentes.`);
                    }

                    // Recargar página después de 2 segundos
                    setTimeout(() => {
                        window.location.reload();
                    }, 2000);
                } else {
                    showAlert('danger', data.mensaje || 'Error al guardar las asistencias.');
                }
            })
            .catch(error => {
                document.getElementById('loadingOverlay').classList.remove('show');
                showAlert('danger', 'Error de conexión. Intente nuevamente.');
                console.error('Error:', error);
            });
        });

        function showAlert(type, message) {
            const alertDiv = document.createElement('div');
            alertDiv.className = `alert alert-${type} alert-dismissible fade show`;
            alertDiv.innerHTML = `
                ${message}
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            `;

            const container = document.querySelector('.card.mb-4');
            container.parentNode.insertBefore(alertDiv, container.nextSibling);

            setTimeout(() => {
                alertDiv.remove();
            }, 5000);
        }

        function actualizarBadge(select, estudianteId) {
            const badge = document.getElementById('badge-' + estudianteId);
            const selectedOption = select.options[select.selectedIndex];

            if (select.value) {
                badge.className = 'estado-badge badge-registrado';
                badge.textContent = '✓';

                // Marcar la fila con borde temporal
                const row = select.closest('tr');
                row.style.background = '#f0fdf4';
                setTimeout(() => {
                    row.style.background = '';
                }, 500);

                cambiosPendientes++;
            } else {
                badge.className = 'estado-badge badge-pendiente';
                badge.textContent = '—';
            }

            actualizarContador();
        }

        function actualizarContador() {
            const contador = document.getElementById('contadorCambios');
            contador.textContent = cambiosPendientes;
        }

        // Reset contador al enviar
        document.getElementById('formAsistencias').addEventListener('submit', function() {
            cambiosPendientes = 0;
            actualizarContador();
        });
    </script>

    <?php include '../includes/layout_scripts.php'; ?>
</body>
</html>
