<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Diagnóstico de Cámara - SICA</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.1/font/bootstrap-icons.css" rel="stylesheet">
    <style>
        body {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            padding: 20px;
        }
        .diagnostic-card {
            background: white;
            border-radius: 20px;
            box-shadow: 0 10px 40px rgba(0,0,0,0.2);
            padding: 30px;
            max-width: 800px;
            margin: 0 auto;
        }
        .test-item {
            padding: 15px;
            margin: 10px 0;
            border-radius: 10px;
            border-left: 4px solid #dee2e6;
        }
        .test-item.pass {
            background: #d1fae5;
            border-left-color: #10b981;
        }
        .test-item.fail {
            background: #fee2e2;
            border-left-color: #ef4444;
        }
        .test-item.warning {
            background: #fef3c7;
            border-left-color: #f59e0b;
        }
        #videoPreview {
            width: 100%;
            max-width: 400px;
            border-radius: 15px;
            margin: 20px auto;
            display: block;
        }
    </style>
</head>
<body>
    <div class="diagnostic-card">
        <div class="text-center mb-4">
            <h1><i class="bi bi-camera-video"></i> Diagnóstico de Cámara</h1>
            <p class="text-muted">Verificación de permisos y acceso a la cámara para escáner QR</p>
        </div>

        <div id="results"></div>

        <div class="text-center mt-4">
            <button onclick="iniciarPruebas()" class="btn btn-primary btn-lg">
                <i class="bi bi-play-circle"></i> Iniciar Diagnóstico
            </button>
        </div>

        <div class="mt-4" style="display: none;" id="videoContainer">
            <h5 class="text-center">Vista Previa de Cámara</h5>
            <video id="videoPreview" autoplay playsinline muted></video>
        </div>

        <div class="mt-4">
            <h6>Acciones Rápidas:</h6>
            <div class="d-flex gap-2 flex-wrap">
                <button onclick="location.href='escanear_qr.php'" class="btn btn-outline-primary">
                    <i class="bi bi-qr-code"></i> Ir a Escáner QR
                </button>
                <button onclick="location.reload()" class="btn btn-outline-secondary">
                    <i class="bi bi-arrow-clockwise"></i> Recargar Página
                </button>
                <button onclick="copiarReporte()" class="btn btn-outline-success">
                    <i class="bi bi-clipboard"></i> Copiar Reporte
                </button>
            </div>
        </div>

        <div class="mt-4 p-3 bg-light rounded">
            <h6><i class="bi bi-info-circle"></i> Información del Sistema</h6>
            <div id="systemInfo"></div>
        </div>
    </div>

    <script>
        const resultados = [];

        function mostrarResultado(titulo, estado, mensaje) {
            const iconos = {
                'pass': 'check-circle-fill text-success',
                'fail': 'x-circle-fill text-danger',
                'warning': 'exclamation-triangle-fill text-warning'
            };

            resultados.push({ titulo, estado, mensaje });

            const html = `
                <div class="test-item ${estado}">
                    <div class="d-flex align-items-start">
                        <i class="bi bi-${iconos[estado]} fs-4 me-3"></i>
                        <div>
                            <h6 class="mb-1">${titulo}</h6>
                            <small class="text-muted">${mensaje}</small>
                        </div>
                    </div>
                </div>
            `;

            document.getElementById('results').innerHTML += html;
        }

        async function iniciarPruebas() {
            document.getElementById('results').innerHTML = '';
            resultados.length = 0;

            // TEST 1: Protocolo HTTPS
            const isHTTPS = location.protocol === 'https:';
            const isLocalhost = location.hostname === 'localhost' || location.hostname === '127.0.0.1';

            if (isHTTPS || isLocalhost) {
                mostrarResultado('Protocolo Seguro', 'pass',
                    `Usando ${location.protocol} (${isLocalhost ? 'localhost permitido' : 'HTTPS activo'})`);
            } else {
                mostrarResultado('Protocolo Seguro', 'fail',
                    `Usando ${location.protocol} - La cámara NO funcionará sin HTTPS o localhost`);
            }

            // TEST 2: Soporte de MediaDevices
            if (navigator.mediaDevices && navigator.mediaDevices.getUserMedia) {
                mostrarResultado('API de Cámara', 'pass',
                    'El navegador soporta navigator.mediaDevices.getUserMedia');
            } else {
                mostrarResultado('API de Cámara', 'fail',
                    'El navegador NO soporta acceso a la cámara. Use Chrome, Firefox, Edge o Safari');
                return;
            }

            // TEST 3: Permisos de Cámara
            try {
                const stream = await navigator.mediaDevices.getUserMedia({ video: true });
                mostrarResultado('Permisos de Cámara', 'pass',
                    'Permisos concedidos correctamente');

                // Mostrar vista previa
                const video = document.getElementById('videoPreview');
                video.srcObject = stream;
                document.getElementById('videoContainer').style.display = 'block';

                // TEST 4: Información de Cámara
                const track = stream.getVideoTracks()[0];
                const settings = track.getSettings();
                mostrarResultado('Cámara Detectada', 'pass',
                    `Resolución: ${settings.width?.toFixed(0)}x${settings.height?.toFixed(0)} - ` +
                    `ID: ${track.label || 'Cámara predeterminada'}`);

                // TEST 5: Calidad del stream
                if (settings.width >= 640 && settings.height >= 480) {
                    mostrarResultado('Calidad de Video', 'pass',
                        `Calidad adecuada (${settings.width}x${settings.height})`);
                } else {
                    mostrarResultado('Calidad de Video', 'warning',
                        `Calidad baja (${settings.width}x${settings.height}) - Podría afectar el escaneo QR`);
                }

                // No detener el stream para que el usuario vea la vista previa
                // stream.getTracks().forEach(track => track.stop());

            } catch (err) {
                if (err.name === 'NotAllowedError' || err.name === 'PermissionDeniedError') {
                    mostrarResultado('Permisos de Cámara', 'fail',
                        'Permiso DENEGADO - Debe permitir el acceso a la cámara en el navegador');
                } else if (err.name === 'NotFoundError' || err.name === 'DevicesNotFoundError') {
                    mostrarResultado('Cámara Detectada', 'fail',
                        'No se encontró ninguna cámara en el dispositivo');
                } else if (err.name === 'NotReadableError' || err.name === 'TrackStartError') {
                    mostrarResultado('Cámara Detectada', 'fail',
                        'La cámara está en uso por otra aplicación o no se puede acceder');
                } else {
                    mostrarResultado('Permisos de Cámara', 'fail',
                        `Error: ${err.name} - ${err.message}`);
                }
            }

            // Información del sistema
            document.getElementById('systemInfo').innerHTML = `
                <ul class="mb-0">
                    <li><strong>Navegador:</strong> ${navigator.userAgent}</li>
                    <li><strong>Protocolo:</strong> ${location.protocol}</li>
                    <li><strong>Hostname:</strong> ${location.hostname}</li>
                    <li><strong>Plataforma:</strong> ${navigator.platform}</li>
                    <li><strong>Cookies:</strong> ${navigator.cookieEnabled ? 'Habilitadas' : 'Deshabilitadas'}</li>
                </ul>
            `;
        }

        function copiarReporte() {
            let reporte = "=== DIAGNÓSTICO DE CÁMARA - SICA ===\n\n";
            resultados.forEach(r => {
                reporte += `[${r.estado.toUpperCase()}] ${r.titulo}\n${r.mensaje}\n\n`;
            });

            navigator.clipboard.writeText(reporte).then(() => {
                alert('Reporte copiado al portapapeles');
            });
        }

        // Cargar información del sistema al inicio
        document.getElementById('systemInfo').innerHTML = `
            <ul class="mb-0">
                <li><strong>Navegador:</strong> ${navigator.userAgent}</li>
                <li><strong>Protocolo:</strong> ${location.protocol}</li>
                <li><strong>Hostname:</strong> ${location.hostname}</li>
                <li><strong>Plataforma:</strong> ${navigator.platform}</li>
            </ul>
        `;
    </script>
</body>
</html>
