<?php
require_once '../config/db.php';
require_once '../config/functions.php';

header('Content-Type: application/json');

requerirRol(['admin', 'auxiliar', 'docente']);

$action = $_GET['action'] ?? $_POST['action'] ?? '';

// Guardar asistencias (registro grupal)
if ($action === 'guardar_asistencias') {
    // Obtener año lectivo seleccionado
    $anio_seleccionado_id = getAnioLectivoSeleccionado();

    if (!$anio_seleccionado_id) {
        echo json_encode([
            'success' => false,
            'mensaje' => 'No hay año lectivo seleccionado'
        ]);
        exit;
    }

    // Obtener datos del formulario
    $fecha = $_POST['fecha'] ?? date('Y-m-d');
    $asistencias = $_POST['asistencias'] ?? [];

    if (empty($asistencias)) {
        echo json_encode([
            'success' => false,
            'mensaje' => 'No se enviaron asistencias para registrar'
        ]);
        exit;
    }

    // Validar fecha
    if (!preg_match('/^\d{4}-\d{2}-\d{2}$/', $fecha)) {
        echo json_encode([
            'success' => false,
            'mensaje' => 'Formato de fecha inválido'
        ]);
        exit;
    }

    // Iniciar transacción
    $pdo->beginTransaction();

    try {
        $nuevas = 0;
        $actualizadas = 0;
        $errores = [];

        foreach ($asistencias as $id_estudiante => $datos) {
            $hora_registro = $datos['hora'] ?? null;
            $id_estado = $datos['estado'] ?? null;
            $id_programacion = $datos['id_programacion'] ?? null;
            $id_turno = $datos['id_turno'] ?? null;

            // Validar campos obligatorios
            if (!$hora_registro || !$id_estado || !$id_programacion || !$id_turno) {
                $errores[] = "Estudiante ID {$id_estudiante}: Faltan datos obligatorios";
                continue;
            }

            // Verificar si ya existe asistencia
            $asistencia_existente = query("
                SELECT id, estado_codigo
                FROM asistencias
                WHERE id_estudiante = ?
                  AND id_anio_lectivo = ?
                  AND fecha = ?
            ", [$id_estudiante, $anio_seleccionado_id, $fecha])->fetch();

            // Obtener datos del estado de asistencia
            $estado_info = query("
                SELECT id, nombre, abreviatura, conteo_asistencia, permite_tardanza
                FROM estados_asistencia
                WHERE id = ? AND estado = 'activo'
            ", [$id_estado])->fetch();

            if (!$estado_info) {
                $errores[] = "Estudiante ID {$id_estudiante}: Estado de asistencia no válido";
                continue;
            }

            // Obtener datos del turno
            $turno_info = query("
                SELECT hora_ingreso, tolerancia_minutos
                FROM turnos
                WHERE id = ? AND estado = 'activo'
            ", [$id_turno])->fetch();

            if (!$turno_info) {
                $errores[] = "Estudiante ID {$id_estudiante}: Turno no válido";
                continue;
            }

            // Calcular minutos de tardanza si corresponde
            $minutos_tardanza = 0;
            if ($estado_info['abreviatura'] === 'TAR' || $estado_info['abreviatura'] === 'TJU') {
                $hora_ingreso = $turno_info['hora_ingreso'];
                $tolerancia = $turno_info['tolerancia_minutos'];

                // Convertir a minutos desde medianoche
                $hora_ingreso_minutos = (int)substr($hora_ingreso, 0, 2) * 60 + (int)substr($hora_ingreso, 3, 2);
                $hora_registro_minutos = (int)substr($hora_registro, 0, 2) * 60 + (int)substr($hora_registro, 3, 2);

                // Calcular tardanza
                if ($hora_registro_minutos > ($hora_ingreso_minutos + $tolerancia)) {
                    $minutos_tardanza = $hora_registro_minutos - $hora_ingreso_minutos;
                }
            }

            // Obtener nombre de usuario
            $usuario = $_SESSION['usuario'] ?? 'SISTEMA';

            if ($asistencia_existente) {
                // Actualizar asistencia existente
                $update = query("
                    UPDATE asistencias
                    SET id_estado_asistencia = ?,
                        estado_codigo = ?,
                        hora_registro = ?,
                        minutos_tardanza = ?,
                        id_programacion = ?,
                        id_turno = ?,
                        metodo_registro = 'manual',
                        registrado_por = ?
                    WHERE id = ?
                ", [
                    $id_estado,
                    $estado_info['abreviatura'],
                    $hora_registro,
                    $minutos_tardanza,
                    $id_programacion,
                    $id_turno,
                    $usuario,
                    $asistencia_existente['id']
                ]);

                if ($update) {
                    $actualizadas++;
                } else {
                    $errores[] = "Estudiante ID {$id_estudiante}: Error al actualizar";
                }
            } else {
                // Insertar nueva asistencia
                $insert = query("
                    INSERT INTO asistencias (
                        id_estudiante,
                        id_anio_lectivo,
                        id_estado_asistencia,
                        estado_codigo,
                        id_programacion,
                        id_turno,
                        fecha,
                        hora_registro,
                        minutos_tardanza,
                        metodo_registro,
                        registrado_por,
                        created_at
                    ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, 'manual', ?, CURRENT_TIMESTAMP)
                ", [
                    $id_estudiante,
                    $anio_seleccionado_id,
                    $id_estado,
                    $estado_info['abreviatura'],
                    $id_programacion,
                    $id_turno,
                    $fecha,
                    $hora_registro,
                    $minutos_tardanza,
                    $usuario
                ]);

                if ($insert) {
                    $nuevas++;
                } else {
                    $errores[] = "Estudiante ID {$id_estudiante}: Error al insertar";
                }
            }
        }

        // Confirmar transacción
        $pdo->commit();

        // Preparar respuesta
        $mensaje = '';
        if ($nuevas > 0 && $actualizadas > 0) {
            $mensaje = "Se registraron {$nuevas} nuevas asistencias y se actualizaron {$actualizadas} existentes.";
        } elseif ($nuevas > 0) {
            $mensaje = "Se registraron {$nuevas} nuevas asistencias correctamente.";
        } elseif ($actualizadas > 0) {
            $mensaje = "Se actualizaron {$actualizadas} asistencias correctamente.";
        } else {
            $mensaje = "No se realizaron cambios.";
        }

        if (!empty($errores)) {
            $mensaje .= " Errores: " . implode('; ', array_slice($errores, 0, 3));
            if (count($errores) > 3) {
                $mensaje .= " y " . (count($errores) - 3) . " errores más.";
            }
        }

        echo json_encode([
            'success' => true,
            'mensaje' => $mensaje,
            'nuevas' => $nuevas,
            'actualizadas' => $actualizadas,
            'errores' => $errores
        ]);

    } catch (Exception $e) {
        // Revertir transacción
        $pdo->rollBack();

        echo json_encode([
            'success' => false,
            'mensaje' => 'Error al procesar: ' . $e->getMessage()
        ]);
    }

    exit;
}

// Obtener historial de asistencias de un estudiante
if ($action === 'obtener_historial') {
    $dni = $_GET['dni'] ?? '';
    $limite = isset($_GET['limite']) ? (int)$_GET['limite'] : 10;

    if (empty($dni)) {
        echo json_encode([
            'success' => false,
            'mensaje' => 'DNI no proporcionado'
        ]);
        exit;
    }

    // Obtener ID del estudiante
    $estudiante = query("SELECT id FROM estudiantes WHERE dni = ? AND estado = 'activo'", [$dni])->fetch();

    if (!$estudiante) {
        echo json_encode([
            'success' => false,
            'mensaje' => 'Estudiante no encontrado'
        ]);
        exit;
    }

    $id_estudiante = $estudiante['id'];
    $anio_seleccionado_id = getAnioLectivoSeleccionado();

    // Obtener últimas asistencias
    $asistencias = query("
        SELECT
            a.id,
            a.fecha,
            a.hora_registro,
            a.minutos_tardanza,
            a.observaciones,
            a.registrado_por,
            a.metodo_registro,
            a.id_estado_asistencia,
            a.estado_codigo,
            ea.nombre as estado_nombre,
            ea.abreviatura as estado_abbr
        FROM asistencias a
        INNER JOIN estados_asistencia ea ON a.id_estado_asistencia = ea.id
        WHERE a.id_estudiante = ?
          AND a.id_anio_lectivo = ?
        ORDER BY a.fecha DESC, a.hora_registro DESC
        LIMIT ?
    ", [$id_estudiante, $anio_seleccionado_id, $limite])->fetchAll();

    echo json_encode([
        'success' => true,
        'asistencias' => $asistencias,
        'total' => count($asistencias)
    ]);
    exit;
}

// Editar asistencia
if ($action === 'editar_asistencia') {
    $id = $_POST['id'] ?? '';
    $fecha = $_POST['fecha'] ?? '';
    $hora = $_POST['hora'] ?? '';
    $id_estado = $_POST['id_estado'] ?? '';
    $observaciones = $_POST['observaciones'] ?? '';

    // Validaciones
    if (!$id || !$fecha || !$hora || !$id_estado) {
        echo json_encode([
            'success' => false,
            'mensaje' => 'Faltan datos obligatorios'
        ]);
        exit;
    }

    try {
        $pdo->beginTransaction();

        // Obtener datos actuales de la asistencia
        $asistencia_actual = query("
            SELECT
                a.*,
                t.hora_ingreso,
                t.tolerancia_minutos
            FROM asistencias a
            INNER JOIN turnos t ON a.id_turno = t.id
            WHERE a.id = ?
        ", [$id])->fetch();

        if (!$asistencia_actual) {
            throw new Exception('Asistencia no encontrada');
        }

        // Obtener datos del nuevo estado
        $estado_info = query("
            SELECT id, nombre, abreviatura, conteo_asistencia, permite_tardanza
            FROM estados_asistencia
            WHERE id = ? AND estado = 'activo'
        ", [$id_estado])->fetch();

        if (!$estado_info) {
            throw new Exception('Estado de asistencia no válido');
        }

        // Calcular minutos de tardanza si corresponde
        $minutos_tardanza = 0;
        if ($estado_info['abreviatura'] === 'TAR' || $estado_info['abreviatura'] === 'TJU') {
            $hora_ingreso = $asistencia_actual['hora_ingreso'];
            $tolerancia = $asistencia_actual['tolerancia_minutos'];

            // Convertir a minutos desde medianoche
            $hora_ingreso_minutos = (int)substr($hora_ingreso, 0, 2) * 60 + (int)substr($hora_ingreso, 3, 2);
            $hora_registro_minutos = (int)substr($hora, 0, 2) * 60 + (int)substr($hora, 3, 2);

            // Calcular tardanza
            if ($hora_registro_minutos > ($hora_ingreso_minutos + $tolerancia)) {
                $minutos_tardanza = $hora_registro_minutos - $hora_ingreso_minutos;
            }
        }

        // Obtener nombre de usuario
        $usuario = $_SESSION['usuario'] ?? 'SISTEMA';

        // Actualizar asistencia
        $update = query("
            UPDATE asistencias
            SET fecha = ?,
                hora_registro = ?,
                id_estado_asistencia = ?,
                estado_codigo = ?,
                minutos_tardanza = ?,
                observaciones = ?,
                registrado_por = ?
            WHERE id = ?
        ", [
            $fecha,
            $hora,
            $id_estado,
            $estado_info['abreviatura'],
            $minutos_tardanza,
            $observaciones,
            $usuario,
            $id
        ]);

        if (!$update) {
            throw new Exception('Error al actualizar la asistencia');
        }

        $pdo->commit();

        echo json_encode([
            'success' => true,
            'mensaje' => 'Asistencia actualizada correctamente',
            'minutos_tardanza' => $minutos_tardanza
        ]);

    } catch (Exception $e) {
        $pdo->rollBack();

        echo json_encode([
            'success' => false,
            'mensaje' => 'Error al actualizar: ' . $e->getMessage()
        ]);
    }
    exit;
}

// Eliminar asistencia
if ($action === 'eliminar_asistencia') {
    $id = $_POST['id'] ?? '';

    if (!$id) {
        echo json_encode([
            'success' => false,
            'mensaje' => 'ID no proporcionado'
        ]);
        exit;
    }

    try {
        // Verificar que la asistencia existe
        $asistencia = query("SELECT id FROM asistencias WHERE id = ?", [$id])->fetch();

        if (!$asistencia) {
            throw new Exception('Asistencia no encontrada');
        }

        // Eliminar asistencia
        $delete = query("DELETE FROM asistencias WHERE id = ?", [$id]);

        if (!$delete) {
            throw new Exception('Error al eliminar la asistencia');
        }

        echo json_encode([
            'success' => true,
            'mensaje' => 'Asistencia eliminada correctamente'
        ]);

    } catch (Exception $e) {
        echo json_encode([
            'success' => false,
            'mensaje' => 'Error al eliminar: ' . $e->getMessage()
        ]);
    }
    exit;
}

// Acción no válida
echo json_encode([
    'success' => false,
    'mensaje' => 'Acción no válida'
]);
?>
