<?php
require_once '../config/db.php';
require_once '../config/functions.php';

requerirRol(['admin']);

$id_apoderado = intval($_GET['id'] ?? 0);

if (!$id_apoderado) {
    header('Location: index.php');
    exit;
}

// Obtener apoderado
$apoderado = query("SELECT a.*, u.usuario
                   FROM apoderados a
                   LEFT JOIN usuarios u ON a.id_usuario = u.id
                   WHERE a.id = :id", ['id' => $id_apoderado])->fetch();

if (!$apoderado) {
    header('Location: index.php');
    exit;
}

$mensaje = '';
$tipo_mensaje = '';

// Obtener año lectivo seleccionado
$anio_seleccionado = getAnioLectivoSeleccionado();

// Procesar vinculación
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['accion'])) {
    if ($_POST['accion'] === 'vincular' && !empty($_POST['id_estudiante'])) {
        try {
            $id_estudiante = intval($_POST['id_estudiante']);
            $parentesco = trim($_POST['parentesco'] ?? 'Padre/Madre');

            // Verificar que el estudiante exista
            $estudiante = query("SELECT id, nombres, apellido_paterno, apellido_materno
                               FROM estudiantes WHERE id = :id", ['id' => $id_estudiante])->fetch();

            if (!$estudiante) {
                throw new Exception('El estudiante no existe');
            }

            // Verificar que no esté vinculado ya en este año lectivo
            $check = query("SELECT COUNT(*) as c FROM estudiantes_apoderados
                          WHERE id_estudiante = :est AND id_anio_lectivo = :anio",
                          ['est' => $id_estudiante, 'anio' => $anio_seleccionado])->fetch();

            if ($check['c'] > 0) {
                throw new Exception('Este estudiante ya tiene un apoderado asignado en este año lectivo');
            }

            // Vincular
            query("INSERT INTO estudiantes_apoderados (id_estudiante, id_apoderado, id_anio_lectivo, parentesco)
                  VALUES (:est, :apod, :anio, :parent)",
                  ['est' => $id_estudiante, 'apod' => $id_apoderado, 'anio' => $anio_seleccionado, 'parent' => $parentesco]);

            $mensaje = 'Estudiante vinculado exitosamente';
            $tipo_mensaje = 'success';
        } catch (Exception $e) {
            $mensaje = 'Error al vincular: ' . $e->getMessage();
            $tipo_mensaje = 'danger';
        }
    } elseif ($_POST['accion'] === 'desvincular' && !empty($_POST['id_vinculacion'])) {
        try {
            $id_vinculacion = intval($_POST['id_vinculacion']);
            query("DELETE FROM estudiantes_apoderados WHERE id = :id", ['id' => $id_vinculacion]);
            $mensaje = 'Vinculación eliminada exitosamente';
            $tipo_mensaje = 'success';
        } catch (Exception $e) {
            $mensaje = 'Error al desvincular: ' . $e->getMessage();
            $tipo_mensaje = 'danger';
        }
    }
}

// Obtener estudiantes vinculados en este año lectivo
$estudiantes_vinculados = query("SELECT ea.*, e.dni, e.apellido_paterno, e.apellido_materno, e.nombres
                                FROM estudiantes_apoderados ea
                                INNER JOIN estudiantes e ON ea.id_estudiante = e.id
                                WHERE ea.id_apoderado = :apod AND ea.id_anio_lectivo = :anio
                                ORDER BY e.apellido_paterno, e.apellido_materno",
                                ['apod' => $id_apoderado, 'anio' => $anio_seleccionado])->fetchAll();

// NOTA: Ya NO cargamos todos los estudiantes disponibles
// En su lugar usamos búsqueda AJAX con autocompletado
// Esto mejora significativamente el rendimiento con 2000+ estudiantes

// Obtener info del año lectivo
$info_anio = query("SELECT anio FROM anios_lectivos WHERE id = :id", ['id' => $anio_seleccionado])->fetch();
?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0, maximum-scale=1.0, user-scalable=no">
    <title>SICA - Vincular Estudiantes</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.1/font/bootstrap-icons.css" rel="stylesheet">
    <link href="../assets/css/sica.css?v=5.1" rel="stylesheet">
    <style>
        #resultados_busqueda {
            background-color: white;
            border: 1px solid #dee2e6;
            border-radius: 0.375rem;
        }
        #resultados_busqueda .list-group-item {
            border: none;
            border-bottom: 1px solid #dee2e6;
        }
        #resultados_busqueda .list-group-item:last-child {
            border-bottom: none;
        }
        #resultados_busqueda .list-group-item:hover {
            background-color: #f8f9fa;
        }
    </style>
</head>
<body>
    <nav class="sidebar">
        <?php include '../includes/sidebar.php'; ?>
    </nav>

    <div class="main-content">
        <div class="d-flex justify-content-between align-items-center mb-4">
            <div>
                <h3 class="mb-1">Vincular Estudiantes</h3>
                <p class="text-muted mb-0">
                    Apoderado: <strong><?= htmlspecialchars($apoderado['apellido_paterno'] . ' ' . $apoderado['apellido_materno'] . ', ' . $apoderado['nombres']) ?></strong>
                    | Año: <?= htmlspecialchars($info_anio['anio']) ?>
                </p>
            </div>
            <a href="ver.php?id=<?= $id_apoderado ?>" class="btn btn-outline-secondary">
                <i class="bi bi-arrow-left"></i> Volver
            </a>
        </div>

        <?php if ($mensaje): ?>
            <div class="alert alert-<?= $tipo_mensaje ?> alert-dismissible fade show" role="alert">
                <?= htmlspecialchars($mensaje) ?>
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            </div>
        <?php endif; ?>

        <div class="row">
            <!-- Formulario de vinculación -->
            <div class="col-lg-5 mb-4">
                <div class="card">
                    <div class="card-header bg-success text-white">
                        <h5 class="mb-0"><i class="bi bi-person-plus"></i> Vincular Nuevo Estudiante</h5>
                    </div>
                    <div class="card-body">
                        <form method="POST" id="formVincular">
                            <input type="hidden" name="accion" value="vincular">
                            <input type="hidden" id="id_estudiante" name="id_estudiante" required>

                            <div class="mb-3 position-relative">
                                <label for="busqueda_estudiante" class="form-label">
                                    Buscar Estudiante <span class="text-danger">*</span>
                                </label>
                                <div class="input-group">
                                    <span class="input-group-text">
                                        <i class="bi bi-search"></i>
                                    </span>
                                    <input type="text" class="form-control" id="busqueda_estudiante"
                                        placeholder="Escriba DNI, apellidos o nombres (mínimo 3 caracteres)"
                                        autocomplete="off">
                                </div>
                                <div id="resultados_busqueda" class="position-absolute w-100 mt-1 shadow-lg"
                                    style="z-index: 1000; max-height: 300px; overflow-y: auto; display: none;">
                                </div>
                                <div id="info_estudiante_seleccionado" class="alert alert-info mt-2" style="display: none;">
                                    <strong>Estudiante seleccionado:</strong>
                                    <span id="nombre_estudiante"></span>
                                </div>
                                <div class="form-text">
                                    <i class="bi bi-lightning-charge text-warning"></i>
                                    Escriba para buscar automáticamente (DNI, apellidos o nombres)
                                </div>
                            </div>

                            <div class="mb-3">
                                <label for="parentesco" class="form-label">Parentesco</label>
                                <input type="text" class="form-control" id="parentesco" name="parentesco"
                                    value="Padre/Madre" placeholder="Ej: Padre, Madre, Tutor, etc.">
                            </div>

                            <button type="submit" class="btn btn-success w-100" id="btnVincular" disabled>
                                <i class="bi bi-link-45deg"></i> Vincular Estudiante
                            </button>
                        </form>
                    </div>
                </div>

                <div class="card mt-3">
                    <div class="card-body">
                        <h6 class="card-title"><i class="bi bi-info-circle text-primary"></i> Información</h6>
                        <ul class="small text-muted mb-0">
                            <li><strong>Búsqueda inteligente:</strong> Escriba DNI, apellidos o nombres (mínimo 3 caracteres)</li>
                            <li>Los resultados aparecen automáticamente mientras escribe</li>
                            <li>Un estudiante solo puede tener un apoderado por año lectivo</li>
                            <li>Los estudiantes ya vinculados no aparecerán en los resultados</li>
                            <li>La vinculación es por año lectivo</li>
                        </ul>
                    </div>
                </div>
            </div>

            <!-- Lista de estudiantes vinculados -->
            <div class="col-lg-7">
                <div class="card">
                    <div class="card-header bg-primary text-white d-flex justify-content-between align-items-center">
                        <h5 class="mb-0"><i class="bi bi-people-fill"></i> Estudiantes Vinculados (<?= count($estudiantes_vinculados) ?>)</h5>
                    </div>
                    <div class="card-body p-0">
                        <?php if (empty($estudiantes_vinculados)): ?>
                            <div class="text-center py-5">
                                <i class="bi bi-person-x text-muted" style="font-size: 3rem;"></i>
                                <p class="text-muted mt-3">No hay estudiantes vinculados en este año lectivo</p>
                            </div>
                        <?php else: ?>
                            <div class="table-responsive">
                                <table class="table table-hover mb-0">
                                    <thead class="table-light">
                                        <tr>
                                            <th>DNI</th>
                                            <th>Estudiante</th>
                                            <th>Parentesco</th>
                                            <th class="text-center">Acciones</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php foreach ($estudiantes_vinculados as $vinculacion): ?>
                                            <tr>
                                                <td><span class="fw-bold"><?= htmlspecialchars($vinculacion['dni']) ?></span></td>
                                                <td>
                                                    <div class="d-flex align-items-center">
                                                        <div class="avatar-circle avatar-sm me-2">
                                                            <?= substr($vinculacion['nombres'], 0, 1) . substr($vinculacion['apellido_paterno'], 0, 1) ?>
                                                        </div>
                                                        <div>
                                                            <div class="fw-bold">
                                                                <?= htmlspecialchars($vinculacion['apellido_paterno'] . ' ' . $vinculacion['apellido_materno'] . ', ' . $vinculacion['nombres']) ?>
                                                            </div>
                                                        </div>
                                                    </div>
                                                </td>
                                                <td>
                                                    <span class="badge bg-info">
                                                        <?= htmlspecialchars($vinculacion['parentesco']) ?>
                                                    </span>
                                                </td>
                                                <td class="text-center">
                                                    <form method="POST" class="d-inline">
                                                        <input type="hidden" name="accion" value="desvincular">
                                                        <input type="hidden" name="id_vinculacion" value="<?= $vinculacion['id'] ?>">
                                                        <button type="submit" class="btn btn-sm btn-outline-danger"
                                                            onclick="return confirm('¿Está seguro de desvincular a este estudiante?')"
                                                            title="Desvincular">
                                                            <i class="bi bi-link-broken"></i>
                                                        </button>
                                                    </form>
                                                </td>
                                            </tr>
                                        <?php endforeach; ?>
                                    </tbody>
                                </table>
                            </div>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        let timeoutBusqueda = null;

        document.getElementById('busqueda_estudiante').addEventListener('input', function() {
            const busqueda = this.value.trim();
            const resultadosDiv = document.getElementById('resultados_busqueda');

            // Limpiar timeout anterior (debounce)
            if (timeoutBusqueda) {
                clearTimeout(timeoutBusqueda);
            }

            // Ocultar resultados si está vacío
            if (busqueda.length < 3) {
                resultadosDiv.style.display = 'none';
                return;
            }

            // Esperar 300ms antes de buscar (debounce)
            timeoutBusqueda = setTimeout(function() {
                console.log('Buscando:', busqueda);

                fetch('ajax_buscar_estudiantes.php?busqueda=' + encodeURIComponent(busqueda))
                    .then(response => {
                        console.log('Status:', response.status);
                        return response.json();
                    })
                    .then(data => {
                        console.log('Datos recibidos:', data);

                        if (data.error) {
                            console.error('Error del servidor:', data.error);
                            resultadosDiv.innerHTML = `
                                <div class="list-group-item text-danger">
                                    <i class="bi bi-exclamation-triangle"></i> ${data.error}
                                </div>
                            `;
                            resultadosDiv.style.display = 'block';
                            resultadosDiv.classList.add('list-group', 'border', 'rounded');
                            return;
                        }

                        if (data.estudiantes && data.estudiantes.length > 0) {
                            resultadosDiv.innerHTML = '';
                            data.estudiantes.forEach(est => {
                                const item = document.createElement('div');
                                item.className = 'list-group-item list-group-item-action';
                                item.style.cursor = 'pointer';
                                item.innerHTML = `
                                    <div class="d-flex justify-content-between align-items-center">
                                        <div>
                                            <strong>${est.apellido_paterno} ${est.apellido_materno}, ${est.nombres}</strong>
                                            <br>
                                            <small class="text-muted">DNI: ${est.dni}</small>
                                        </div>
                                        <i class="bi bi-person-plus text-success"></i>
                                    </div>
                                `;
                                item.addEventListener('click', function() {
                                    seleccionarEstudiante(est);
                                });
                                resultadosDiv.appendChild(item);
                            });
                            resultadosDiv.style.display = 'block';
                            resultadosDiv.classList.add('list-group', 'border', 'rounded');
                        } else {
                            resultadosDiv.innerHTML = `
                                <div class="list-group-item text-muted">
                                    <i class="bi bi-search"></i> No se encontraron estudiantes
                                </div>
                            `;
                            resultadosDiv.style.display = 'block';
                            resultadosDiv.classList.add('list-group', 'border', 'rounded');
                        }
                    })
                    .catch(error => {
                        console.error('Error de red:', error);
                        resultadosDiv.innerHTML = `
                            <div class="list-group-item text-danger">
                                <i class="bi bi-exclamation-triangle"></i> Error de conexión
                            </div>
                        `;
                        resultadosDiv.style.display = 'block';
                    });
            }, 300);
        });

        function seleccionarEstudiante(estudiante) {
            document.getElementById('id_estudiante').value = estudiante.id;
            document.getElementById('busqueda_estudiante').value =
                `${estudiante.apellido_paterno} ${estudiante.apellido_materno}, ${estudiante.nombres}`;
            document.getElementById('resultados_busqueda').style.display = 'none';
            document.getElementById('info_estudiante_seleccionado').style.display = 'block';
            document.getElementById('nombre_estudiante').textContent =
                `${estudiante.apellido_paterno} ${estudiante.apellido_materno}, ${estudiante.nombres} (DNI: ${estudiante.dni})`;
            document.getElementById('btnVincular').disabled = false;
        }

        // Cerrar resultados si hacen clic fuera
        document.addEventListener('click', function(e) {
            const resultadosDiv = document.getElementById('resultados_busqueda');
            const inputBusqueda = document.getElementById('busqueda_estudiante');
            if (!resultadosDiv.contains(e.target) && e.target !== inputBusqueda) {
                resultadosDiv.style.display = 'none';
            }
        });
    </script>
</body>
</html>
